/*
 * Copyright (c) 2022-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef ARM_COMPUTE_ENABLE_SVE

#include "arm_gemm.hpp"
#include "../../utils.hpp"
#include "../../bfloat.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void sve_ffhybrid_fp32bf16fp32_mmla_4x6VL (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<float> A_arg,
    size_t M, size_t N, const bfloat16 *B_ptr, size_t B_stride, IndirectOutputArg<float> output_arg,
    const float *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        float maxval = static_cast<float>(std::numeric_limits<float>::infinity());
        float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const bfloat16 *B_ptr = {};
        const bfloat16 *cur_B_ptr = {};
        size_t B_stride = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
        const float *bias = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    ka.bias = bias;
    ka.B_stride = B_stride;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "ptrue p7.b\n"
      "1:"  // Row loop
      "cmp %x[M], #0x4\n"
      "bge 43f\n"
      "cmp %x[M], #0x2\n"
      "bgt 29f\n"
      "beq 15f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cntw x21, ALL, MUL #5\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x28, x9, x20, LSL #1\n"
      "add x27, x28, x20, LSL #1\n"
      "add x20, x27, x20, LSL #1\n"
      "cmp x14, x21\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 3f\n"
      "decw x21\n"
      "mov x27, x12\n"
      "cmp x14, x21\n"
      "bgt 3f\n"
      "decw x21\n"
      "mov x28, x12\n"
      "cmp x14, x21\n"
      "bgt 3f\n"
      "decw x21\n"
      "mov x9, x12\n"
      "cmp x14, x21\n"
      "bgt 3f\n"
      "decw x21\n"
      "mov x10, x12\n"
      "cmp x14, x21\n"
      "bgt 3f\n"
      "mov x11, x12\n"
      "3:"  // Height 1: B setup done
      "mov x20, #0x0\n"
      "whilelt p6.s, x20, x14\n"
      "incw x20\n"
      "whilelt p5.s, x20, x14\n"
      "incw x20\n"
      "whilelt p4.s, x20, x14\n"
      "incw x20\n"
      "whilelt p3.s, x20, x14\n"
      "incw x20\n"
      "whilelt p2.s, x20, x14\n"
      "incw x20\n"
      "whilelt p1.s, x20, x14\n"
      "cbz x15, 4f\n"
      "ld1w { z8.s }, p7/Z, [x15]\n"
      "ld1w { z9.s }, p7/Z, [x15, #1, MUL VL]\n"
      "ld1w { z10.s }, p7/Z, [x15, #2, MUL VL]\n"
      "ld1w { z11.s }, p7/Z, [x15, #3, MUL VL]\n"
      "ld1w { z12.s }, p7/Z, [x15, #4, MUL VL]\n"
      "ld1w { z13.s }, p7/Z, [x15, #5, MUL VL]\n"
      "addvl x15, x15, #6\n"
      "zip2 z14.d, z8.d, z8.d\n"
      "zip1 z8.d, z8.d, z8.d\n"
      "zip2 z15.d, z9.d, z9.d\n"
      "zip1 z9.d, z9.d, z9.d\n"
      "zip2 z16.d, z10.d, z10.d\n"
      "zip1 z10.d, z10.d, z10.d\n"
      "zip2 z17.d, z11.d, z11.d\n"
      "zip1 z11.d, z11.d, z11.d\n"
      "zip2 z18.d, z12.d, z12.d\n"
      "zip1 z12.d, z12.d, z12.d\n"
      "zip2 z19.d, z13.d, z13.d\n"
      "zip1 z13.d, z13.d, z13.d\n"
      "b 6f\n"
      "4:"  // Height 1: no bias
      "tbz %x[flags], #0, 5f\n"
      "ld1w { z25.s }, p6/Z, [x13]\n"
      "ld1w { z24.s }, p5/Z, [x13, #1, MUL VL]\n"
      "ld1w { z23.s }, p4/Z, [x13, #2, MUL VL]\n"
      "ld1w { z22.s }, p3/Z, [x13, #3, MUL VL]\n"
      "ld1w { z21.s }, p2/Z, [x13, #4, MUL VL]\n"
      "ld1w { z20.s }, p1/Z, [x13, #5, MUL VL]\n"
      "zip1 z8.d, z25.d, z14.d\n"
      "zip2 z14.d, z25.d, z14.d\n"
      "zip1 z9.d, z24.d, z15.d\n"
      "zip2 z15.d, z24.d, z15.d\n"
      "zip1 z10.d, z23.d, z16.d\n"
      "zip2 z16.d, z23.d, z16.d\n"
      "zip1 z11.d, z22.d, z17.d\n"
      "zip2 z17.d, z22.d, z17.d\n"
      "zip1 z12.d, z21.d, z18.d\n"
      "zip2 z18.d, z21.d, z18.d\n"
      "zip1 z13.d, z20.d, z19.d\n"
      "zip2 z19.d, z20.d, z19.d\n"
      "b 6f\n"
      "5:"  // Height 1: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "6:"  // Height 1: setup done
      "mov x26, #0x0\n"
      "7:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 8f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "cbnz x26, 9f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20, LSL #2\n"
      "b 9f\n"
      "8:"  // Height 1: setup direct input
      "mov x24, %x[input_ptr]\n"
      "9:"  // Height 1: input setup done
      "cmp x25, #0x4\n"
      "ble 11f\n"
      "10:"  // Height 1: Multiply loop: Main loop head
      "whilelt p0.s, XZR, x25\n"
      "ld1h { z23.h }, p7/Z, [x12]\n"
      "ld1h { z22.h }, p7/Z, [x12, #1, MUL VL]\n"
      "sub x25, x25, #0x4\n"
      "ld1h { z21.h }, p7/Z, [x11]\n"
      "ld1h { z20.h }, p7/Z, [x11, #1, MUL VL]\n"
      "cmp x25, #0x4\n"
      "addvl x12, x12, #2\n"
      "addvl x11, x11, #2\n"
      "ld1rqw { z24.s }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x658abf18  // bfcvt z24.h, p7/M, z24.s\n"
      "uzp1 z24.h, z24.h, z24.h\n"
      ".inst 0x6477e708  // bfmmla z8.s, z24.h, z23.h\n"
      "ld1h { z23.h }, p7/Z, [x10]\n"
      ".inst 0x6476e70e  // bfmmla z14.s, z24.h, z22.h\n"
      "ld1h { z22.h }, p7/Z, [x10, #1, MUL VL]\n"
      "addvl x10, x10, #2\n"
      ".inst 0x6475e709  // bfmmla z9.s, z24.h, z21.h\n"
      "ld1h { z21.h }, p7/Z, [x9]\n"
      ".inst 0x6474e70f  // bfmmla z15.s, z24.h, z20.h\n"
      "ld1h { z20.h }, p7/Z, [x9, #1, MUL VL]\n"
      "addvl x9, x9, #2\n"
      ".inst 0x6477e70a  // bfmmla z10.s, z24.h, z23.h\n"
      "ld1h { z23.h }, p7/Z, [x28]\n"
      ".inst 0x6476e710  // bfmmla z16.s, z24.h, z22.h\n"
      "ld1h { z22.h }, p7/Z, [x28, #1, MUL VL]\n"
      "addvl x28, x28, #2\n"
      ".inst 0x6475e70b  // bfmmla z11.s, z24.h, z21.h\n"
      "ld1h { z21.h }, p7/Z, [x27]\n"
      ".inst 0x6474e711  // bfmmla z17.s, z24.h, z20.h\n"
      "ld1h { z20.h }, p7/Z, [x27, #1, MUL VL]\n"
      "addvl x27, x27, #2\n"
      ".inst 0x6477e70c  // bfmmla z12.s, z24.h, z23.h\n"
      ".inst 0x6476e712  // bfmmla z18.s, z24.h, z22.h\n"
      ".inst 0x6475e70d  // bfmmla z13.s, z24.h, z21.h\n"
      ".inst 0x6474e713  // bfmmla z19.s, z24.h, z20.h\n"
      "bgt 10b\n"
      "11:"  // Height 1: Multiply loop: Single iteration only
      "whilelt p0.s, XZR, x25\n"
      "ld1h { z23.h }, p7/Z, [x12]\n"
      "ld1h { z22.h }, p7/Z, [x12, #1, MUL VL]\n"
      "addvl x12, x12, #2\n"
      "ld1h { z21.h }, p7/Z, [x11]\n"
      "ld1h { z20.h }, p7/Z, [x11, #1, MUL VL]\n"
      "addvl x11, x11, #2\n"
      "ld1rqw { z24.s }, p0/Z, [x24]\n"
      ".inst 0x658abf18  // bfcvt z24.h, p7/M, z24.s\n"
      "uzp1 z24.h, z24.h, z24.h\n"
      ".inst 0x6477e708  // bfmmla z8.s, z24.h, z23.h\n"
      "ld1h { z23.h }, p7/Z, [x10]\n"
      ".inst 0x6476e70e  // bfmmla z14.s, z24.h, z22.h\n"
      "ld1h { z22.h }, p7/Z, [x10, #1, MUL VL]\n"
      "addvl x10, x10, #2\n"
      ".inst 0x6475e709  // bfmmla z9.s, z24.h, z21.h\n"
      "ld1h { z21.h }, p7/Z, [x9]\n"
      ".inst 0x6474e70f  // bfmmla z15.s, z24.h, z20.h\n"
      "ld1h { z20.h }, p7/Z, [x9, #1, MUL VL]\n"
      "addvl x9, x9, #2\n"
      ".inst 0x6477e70a  // bfmmla z10.s, z24.h, z23.h\n"
      "ld1h { z23.h }, p7/Z, [x28]\n"
      ".inst 0x6476e710  // bfmmla z16.s, z24.h, z22.h\n"
      "ld1h { z22.h }, p7/Z, [x28, #1, MUL VL]\n"
      "addvl x28, x28, #2\n"
      ".inst 0x6475e70b  // bfmmla z11.s, z24.h, z21.h\n"
      "ld1h { z21.h }, p7/Z, [x27]\n"
      ".inst 0x6474e711  // bfmmla z17.s, z24.h, z20.h\n"
      "ld1h { z20.h }, p7/Z, [x27, #1, MUL VL]\n"
      "addvl x27, x27, #2\n"
      ".inst 0x6477e70c  // bfmmla z12.s, z24.h, z23.h\n"
      ".inst 0x6476e712  // bfmmla z18.s, z24.h, z22.h\n"
      ".inst 0x6475e70d  // bfmmla z13.s, z24.h, z21.h\n"
      ".inst 0x6474e713  // bfmmla z19.s, z24.h, z20.h\n"
      "12:"  // Height 1: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 7b\n"
      "uzp1 z8.d, z8.d, z14.d\n"
      "uzp1 z9.d, z9.d, z15.d\n"
      "uzp1 z10.d, z10.d, z16.d\n"
      "uzp1 z11.d, z11.d, z17.d\n"
      "uzp1 z12.d, z12.d, z18.d\n"
      "uzp1 z13.d, z13.d, z19.d\n"
      "tbz %x[flags], #1, 13f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z21.s }, p7/Z, [x21]\n"
      "ld1rw { z20.s }, p7/Z, [x20]\n"
      "fmin z8.s, p7/M, z8.s, z21.s\n"
      "fmin z9.s, p7/M, z9.s, z21.s\n"
      "fmin z10.s, p7/M, z10.s, z21.s\n"
      "fmin z11.s, p7/M, z11.s, z21.s\n"
      "fmin z12.s, p7/M, z12.s, z21.s\n"
      "fmin z13.s, p7/M, z13.s, z21.s\n"
      "fmax z8.s, p7/M, z8.s, z20.s\n"
      "fmax z9.s, p7/M, z9.s, z20.s\n"
      "fmax z10.s, p7/M, z10.s, z20.s\n"
      "fmax z11.s, p7/M, z11.s, z20.s\n"
      "fmax z12.s, p7/M, z12.s, z20.s\n"
      "fmax z13.s, p7/M, z13.s, z20.s\n"
      "13:"  // Height 1: No activation
      "st1w { z8.s }, p6, [x13]\n"
      "st1w { z9.s }, p5, [x13, #1, MUL VL]\n"
      "st1w { z10.s }, p4, [x13, #2, MUL VL]\n"
      "st1w { z11.s }, p3, [x13, #3, MUL VL]\n"
      "st1w { z12.s }, p2, [x13, #4, MUL VL]\n"
      "st1w { z13.s }, p1, [x13, #5, MUL VL]\n"
      "addvl x13, x13, #6\n"
      "14:"  // Height 1: Writeback done
      "decw x14, ALL, MUL #6\n"
      "cmp x14, XZR\n"
      "bgt 2b\n"
      "b 58f\n"
      "15:"  // Height 2
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "16:"  // Height 2: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cntw x21, ALL, MUL #5\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x28, x9, x20, LSL #1\n"
      "add x27, x28, x20, LSL #1\n"
      "add x20, x27, x20, LSL #1\n"
      "cmp x14, x21\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 17f\n"
      "decw x21\n"
      "mov x27, x12\n"
      "cmp x14, x21\n"
      "bgt 17f\n"
      "decw x21\n"
      "mov x28, x12\n"
      "cmp x14, x21\n"
      "bgt 17f\n"
      "decw x21\n"
      "mov x9, x12\n"
      "cmp x14, x21\n"
      "bgt 17f\n"
      "decw x21\n"
      "mov x10, x12\n"
      "cmp x14, x21\n"
      "bgt 17f\n"
      "mov x11, x12\n"
      "17:"  // Height 2: B setup done
      "mov x20, #0x0\n"
      "whilelt p6.s, x20, x14\n"
      "incw x20\n"
      "whilelt p5.s, x20, x14\n"
      "incw x20\n"
      "whilelt p4.s, x20, x14\n"
      "incw x20\n"
      "whilelt p3.s, x20, x14\n"
      "incw x20\n"
      "whilelt p2.s, x20, x14\n"
      "incw x20\n"
      "whilelt p1.s, x20, x14\n"
      "cbz x15, 18f\n"
      "ld1w { z8.s }, p7/Z, [x15]\n"
      "ld1w { z9.s }, p7/Z, [x15, #1, MUL VL]\n"
      "ld1w { z10.s }, p7/Z, [x15, #2, MUL VL]\n"
      "ld1w { z11.s }, p7/Z, [x15, #3, MUL VL]\n"
      "ld1w { z12.s }, p7/Z, [x15, #4, MUL VL]\n"
      "ld1w { z13.s }, p7/Z, [x15, #5, MUL VL]\n"
      "addvl x15, x15, #6\n"
      "zip2 z14.d, z8.d, z8.d\n"
      "zip1 z8.d, z8.d, z8.d\n"
      "zip2 z15.d, z9.d, z9.d\n"
      "zip1 z9.d, z9.d, z9.d\n"
      "zip2 z16.d, z10.d, z10.d\n"
      "zip1 z10.d, z10.d, z10.d\n"
      "zip2 z17.d, z11.d, z11.d\n"
      "zip1 z11.d, z11.d, z11.d\n"
      "zip2 z18.d, z12.d, z12.d\n"
      "zip1 z12.d, z12.d, z12.d\n"
      "zip2 z19.d, z13.d, z13.d\n"
      "zip1 z13.d, z13.d, z13.d\n"
      "b 20f\n"
      "18:"  // Height 2: no bias
      "tbz %x[flags], #0, 19f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z25.s }, p6/Z, [x13]\n"
      "ld1w { z24.s }, p5/Z, [x13, #1, MUL VL]\n"
      "ld1w { z23.s }, p4/Z, [x13, #2, MUL VL]\n"
      "ld1w { z22.s }, p3/Z, [x13, #3, MUL VL]\n"
      "ld1w { z21.s }, p2/Z, [x13, #4, MUL VL]\n"
      "add x20, x13, x20, LSL #2\n"
      "ld1w { z20.s }, p1/Z, [x13, #5, MUL VL]\n"
      "ld1w { z14.s }, p6/Z, [x20]\n"
      "ld1w { z15.s }, p5/Z, [x20, #1, MUL VL]\n"
      "ld1w { z16.s }, p4/Z, [x20, #2, MUL VL]\n"
      "ld1w { z17.s }, p3/Z, [x20, #3, MUL VL]\n"
      "ld1w { z18.s }, p2/Z, [x20, #4, MUL VL]\n"
      "ld1w { z19.s }, p1/Z, [x20, #5, MUL VL]\n"
      "zip1 z8.d, z25.d, z14.d\n"
      "zip2 z14.d, z25.d, z14.d\n"
      "zip1 z9.d, z24.d, z15.d\n"
      "zip2 z15.d, z24.d, z15.d\n"
      "zip1 z10.d, z23.d, z16.d\n"
      "zip2 z16.d, z23.d, z16.d\n"
      "zip1 z11.d, z22.d, z17.d\n"
      "zip2 z17.d, z22.d, z17.d\n"
      "zip1 z12.d, z21.d, z18.d\n"
      "zip2 z18.d, z21.d, z18.d\n"
      "zip1 z13.d, z20.d, z19.d\n"
      "zip2 z19.d, z20.d, z19.d\n"
      "b 20f\n"
      "19:"  // Height 2: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "20:"  // Height 2: setup done
      "mov x26, #0x0\n"
      "21:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 22f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "cbnz x26, 23f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "b 23f\n"
      "22:"  // Height 2: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21, LSL #2\n"
      "23:"  // Height 2: input setup done
      "cmp x25, #0x4\n"
      "ble 25f\n"
      "24:"  // Height 2: Multiply loop: Main loop head
      "whilelt p0.s, XZR, x25\n"
      "ld1h { z23.h }, p7/Z, [x12]\n"
      "ld1h { z22.h }, p7/Z, [x12, #1, MUL VL]\n"
      "sub x25, x25, #0x4\n"
      "ld1h { z21.h }, p7/Z, [x11]\n"
      "ld1h { z25.h }, p7/Z, [x11, #1, MUL VL]\n"
      "cmp x25, #0x4\n"
      "addvl x12, x12, #2\n"
      "addvl x11, x11, #2\n"
      "ld1rqw { z24.s }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "ld1rqw { z20.s }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x658abf18  // bfcvt z24.h, p7/M, z24.s\n"
      ".inst 0x658abe94  // bfcvt z20.h, p7/M, z20.s\n"
      "uzp1 z24.h, z24.h, z24.h\n"
      "uzp1 z20.h, z20.h, z20.h\n"
      "trn1 z24.d, z24.d, z20.d\n"
      ".inst 0x6477e708  // bfmmla z8.s, z24.h, z23.h\n"
      "ld1h { z23.h }, p7/Z, [x10]\n"
      ".inst 0x6476e70e  // bfmmla z14.s, z24.h, z22.h\n"
      "ld1h { z22.h }, p7/Z, [x10, #1, MUL VL]\n"
      "addvl x10, x10, #2\n"
      ".inst 0x6475e709  // bfmmla z9.s, z24.h, z21.h\n"
      "ld1h { z21.h }, p7/Z, [x9]\n"
      ".inst 0x6479e70f  // bfmmla z15.s, z24.h, z25.h\n"
      "ld1h { z20.h }, p7/Z, [x9, #1, MUL VL]\n"
      "addvl x9, x9, #2\n"
      ".inst 0x6477e70a  // bfmmla z10.s, z24.h, z23.h\n"
      "ld1h { z23.h }, p7/Z, [x28]\n"
      ".inst 0x6476e710  // bfmmla z16.s, z24.h, z22.h\n"
      "ld1h { z22.h }, p7/Z, [x28, #1, MUL VL]\n"
      "addvl x28, x28, #2\n"
      ".inst 0x6475e70b  // bfmmla z11.s, z24.h, z21.h\n"
      "ld1h { z21.h }, p7/Z, [x27]\n"
      ".inst 0x6474e711  // bfmmla z17.s, z24.h, z20.h\n"
      "ld1h { z20.h }, p7/Z, [x27, #1, MUL VL]\n"
      "addvl x27, x27, #2\n"
      ".inst 0x6477e70c  // bfmmla z12.s, z24.h, z23.h\n"
      ".inst 0x6476e712  // bfmmla z18.s, z24.h, z22.h\n"
      ".inst 0x6475e70d  // bfmmla z13.s, z24.h, z21.h\n"
      ".inst 0x6474e713  // bfmmla z19.s, z24.h, z20.h\n"
      "bgt 24b\n"
      "25:"  // Height 2: Multiply loop: Single iteration only
      "whilelt p0.s, XZR, x25\n"
      "ld1h { z23.h }, p7/Z, [x12]\n"
      "ld1h { z22.h }, p7/Z, [x12, #1, MUL VL]\n"
      "addvl x12, x12, #2\n"
      "ld1h { z21.h }, p7/Z, [x11]\n"
      "ld1h { z25.h }, p7/Z, [x11, #1, MUL VL]\n"
      "addvl x11, x11, #2\n"
      "ld1rqw { z24.s }, p0/Z, [x24]\n"
      "ld1rqw { z20.s }, p0/Z, [x23]\n"
      ".inst 0x658abf18  // bfcvt z24.h, p7/M, z24.s\n"
      ".inst 0x658abe94  // bfcvt z20.h, p7/M, z20.s\n"
      "uzp1 z24.h, z24.h, z24.h\n"
      "uzp1 z20.h, z20.h, z20.h\n"
      "trn1 z24.d, z24.d, z20.d\n"
      ".inst 0x6477e708  // bfmmla z8.s, z24.h, z23.h\n"
      "ld1h { z23.h }, p7/Z, [x10]\n"
      ".inst 0x6476e70e  // bfmmla z14.s, z24.h, z22.h\n"
      "ld1h { z22.h }, p7/Z, [x10, #1, MUL VL]\n"
      "addvl x10, x10, #2\n"
      ".inst 0x6475e709  // bfmmla z9.s, z24.h, z21.h\n"
      "ld1h { z21.h }, p7/Z, [x9]\n"
      ".inst 0x6479e70f  // bfmmla z15.s, z24.h, z25.h\n"
      "ld1h { z20.h }, p7/Z, [x9, #1, MUL VL]\n"
      "addvl x9, x9, #2\n"
      ".inst 0x6477e70a  // bfmmla z10.s, z24.h, z23.h\n"
      "ld1h { z23.h }, p7/Z, [x28]\n"
      ".inst 0x6476e710  // bfmmla z16.s, z24.h, z22.h\n"
      "ld1h { z22.h }, p7/Z, [x28, #1, MUL VL]\n"
      "addvl x28, x28, #2\n"
      ".inst 0x6475e70b  // bfmmla z11.s, z24.h, z21.h\n"
      "ld1h { z21.h }, p7/Z, [x27]\n"
      ".inst 0x6474e711  // bfmmla z17.s, z24.h, z20.h\n"
      "ld1h { z20.h }, p7/Z, [x27, #1, MUL VL]\n"
      "addvl x27, x27, #2\n"
      ".inst 0x6477e70c  // bfmmla z12.s, z24.h, z23.h\n"
      ".inst 0x6476e712  // bfmmla z18.s, z24.h, z22.h\n"
      ".inst 0x6475e70d  // bfmmla z13.s, z24.h, z21.h\n"
      ".inst 0x6474e713  // bfmmla z19.s, z24.h, z20.h\n"
      "26:"  // Height 2: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 21b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 z4.d, z8.d, z14.d\n"
      "uzp2 z8.d, z8.d, z14.d\n"
      "uzp1 z14.d, z9.d, z15.d\n"
      "uzp2 z9.d, z9.d, z15.d\n"
      "uzp1 z15.d, z10.d, z16.d\n"
      "uzp2 z10.d, z10.d, z16.d\n"
      "uzp1 z16.d, z11.d, z17.d\n"
      "uzp2 z11.d, z11.d, z17.d\n"
      "add x24, x13, x20, LSL #2\n"
      "uzp1 z17.d, z12.d, z18.d\n"
      "uzp2 z12.d, z12.d, z18.d\n"
      "uzp1 z18.d, z13.d, z19.d\n"
      "uzp2 z13.d, z13.d, z19.d\n"
      "tbz %x[flags], #1, 27f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z20.s }, p7/Z, [x21]\n"
      "ld1rw { z19.s }, p7/Z, [x20]\n"
      "fmin z4.s, p7/M, z4.s, z20.s\n"
      "fmin z14.s, p7/M, z14.s, z20.s\n"
      "fmin z15.s, p7/M, z15.s, z20.s\n"
      "fmin z16.s, p7/M, z16.s, z20.s\n"
      "fmin z17.s, p7/M, z17.s, z20.s\n"
      "fmin z18.s, p7/M, z18.s, z20.s\n"
      "fmin z8.s, p7/M, z8.s, z20.s\n"
      "fmin z9.s, p7/M, z9.s, z20.s\n"
      "fmin z10.s, p7/M, z10.s, z20.s\n"
      "fmin z11.s, p7/M, z11.s, z20.s\n"
      "fmin z12.s, p7/M, z12.s, z20.s\n"
      "fmin z13.s, p7/M, z13.s, z20.s\n"
      "fmax z4.s, p7/M, z4.s, z19.s\n"
      "fmax z14.s, p7/M, z14.s, z19.s\n"
      "fmax z15.s, p7/M, z15.s, z19.s\n"
      "fmax z16.s, p7/M, z16.s, z19.s\n"
      "fmax z17.s, p7/M, z17.s, z19.s\n"
      "fmax z18.s, p7/M, z18.s, z19.s\n"
      "fmax z8.s, p7/M, z8.s, z19.s\n"
      "fmax z9.s, p7/M, z9.s, z19.s\n"
      "fmax z10.s, p7/M, z10.s, z19.s\n"
      "fmax z11.s, p7/M, z11.s, z19.s\n"
      "fmax z12.s, p7/M, z12.s, z19.s\n"
      "fmax z13.s, p7/M, z13.s, z19.s\n"
      "27:"  // Height 2: No activation
      "st1w { z4.s }, p6, [x13]\n"
      "st1w { z14.s }, p5, [x13, #1, MUL VL]\n"
      "st1w { z15.s }, p4, [x13, #2, MUL VL]\n"
      "st1w { z16.s }, p3, [x13, #3, MUL VL]\n"
      "st1w { z17.s }, p2, [x13, #4, MUL VL]\n"
      "st1w { z18.s }, p1, [x13, #5, MUL VL]\n"
      "addvl x13, x13, #6\n"
      "st1w { z8.s }, p6, [x24]\n"
      "st1w { z9.s }, p5, [x24, #1, MUL VL]\n"
      "st1w { z10.s }, p4, [x24, #2, MUL VL]\n"
      "st1w { z11.s }, p3, [x24, #3, MUL VL]\n"
      "st1w { z12.s }, p2, [x24, #4, MUL VL]\n"
      "st1w { z13.s }, p1, [x24, #5, MUL VL]\n"
      "28:"  // Height 2: Writeback done
      "decw x14, ALL, MUL #6\n"
      "cmp x14, XZR\n"
      "bgt 16b\n"
      "b 58f\n"
      "29:"  // Height 3
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "30:"  // Height 3: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cntw x21, ALL, MUL #5\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x28, x9, x20, LSL #1\n"
      "add x27, x28, x20, LSL #1\n"
      "add x20, x27, x20, LSL #1\n"
      "cmp x14, x21\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 31f\n"
      "decw x21\n"
      "mov x27, x12\n"
      "cmp x14, x21\n"
      "bgt 31f\n"
      "decw x21\n"
      "mov x28, x12\n"
      "cmp x14, x21\n"
      "bgt 31f\n"
      "decw x21\n"
      "mov x9, x12\n"
      "cmp x14, x21\n"
      "bgt 31f\n"
      "decw x21\n"
      "mov x10, x12\n"
      "cmp x14, x21\n"
      "bgt 31f\n"
      "mov x11, x12\n"
      "31:"  // Height 3: B setup done
      "mov x20, #0x0\n"
      "whilelt p6.s, x20, x14\n"
      "incw x20\n"
      "whilelt p5.s, x20, x14\n"
      "incw x20\n"
      "whilelt p4.s, x20, x14\n"
      "incw x20\n"
      "whilelt p3.s, x20, x14\n"
      "incw x20\n"
      "whilelt p2.s, x20, x14\n"
      "incw x20\n"
      "whilelt p1.s, x20, x14\n"
      "cbz x15, 32f\n"
      "ld1w { z8.s }, p7/Z, [x15]\n"
      "ld1w { z9.s }, p7/Z, [x15, #1, MUL VL]\n"
      "ld1w { z10.s }, p7/Z, [x15, #2, MUL VL]\n"
      "ld1w { z11.s }, p7/Z, [x15, #3, MUL VL]\n"
      "ld1w { z12.s }, p7/Z, [x15, #4, MUL VL]\n"
      "ld1w { z13.s }, p7/Z, [x15, #5, MUL VL]\n"
      "addvl x15, x15, #6\n"
      "zip2 z14.d, z8.d, z8.d\n"
      "zip1 z8.d, z8.d, z8.d\n"
      "zip2 z15.d, z9.d, z9.d\n"
      "zip1 z9.d, z9.d, z9.d\n"
      "zip2 z16.d, z10.d, z10.d\n"
      "zip1 z10.d, z10.d, z10.d\n"
      "zip2 z17.d, z11.d, z11.d\n"
      "zip1 z11.d, z11.d, z11.d\n"
      "zip2 z18.d, z12.d, z12.d\n"
      "zip1 z12.d, z12.d, z12.d\n"
      "zip2 z19.d, z13.d, z13.d\n"
      "zip1 z13.d, z13.d, z13.d\n"
      "mov z20.d, z8.d\n"
      "mov z26.d, z14.d\n"
      "mov z21.d, z9.d\n"
      "mov z27.d, z15.d\n"
      "mov z22.d, z10.d\n"
      "mov z28.d, z16.d\n"
      "mov z23.d, z11.d\n"
      "mov z29.d, z17.d\n"
      "mov z24.d, z12.d\n"
      "mov z30.d, z18.d\n"
      "mov z25.d, z13.d\n"
      "mov z31.d, z19.d\n"
      "b 34f\n"
      "32:"  // Height 3: no bias
      "tbz %x[flags], #0, 33f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z22.s }, p6/Z, [x13]\n"
      "ld1w { z24.s }, p5/Z, [x13, #1, MUL VL]\n"
      "ld1w { z0.s }, p4/Z, [x13, #2, MUL VL]\n"
      "ld1w { z2.s }, p3/Z, [x13, #3, MUL VL]\n"
      "ld1w { z1.s }, p2/Z, [x13, #4, MUL VL]\n"
      "add x21, x13, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z20.s }, p1/Z, [x13, #5, MUL VL]\n"
      "ld1w { z14.s }, p6/Z, [x21]\n"
      "ld1w { z15.s }, p5/Z, [x21, #1, MUL VL]\n"
      "ld1w { z16.s }, p4/Z, [x21, #2, MUL VL]\n"
      "ld1w { z17.s }, p3/Z, [x21, #3, MUL VL]\n"
      "ld1w { z18.s }, p2/Z, [x21, #4, MUL VL]\n"
      "ld1w { z19.s }, p1/Z, [x21, #5, MUL VL]\n"
      "ld1w { z21.s }, p6/Z, [x20]\n"
      "zip1 z8.d, z22.d, z14.d\n"
      "zip2 z14.d, z22.d, z14.d\n"
      "ld1w { z22.s }, p5/Z, [x20, #1, MUL VL]\n"
      "ld1w { z23.s }, p4/Z, [x20, #2, MUL VL]\n"
      "zip1 z9.d, z24.d, z15.d\n"
      "zip2 z15.d, z24.d, z15.d\n"
      "ld1w { z24.s }, p3/Z, [x20, #3, MUL VL]\n"
      "ld1w { z25.s }, p2/Z, [x20, #4, MUL VL]\n"
      "zip1 z10.d, z0.d, z16.d\n"
      "zip2 z16.d, z0.d, z16.d\n"
      "ld1w { z0.s }, p1/Z, [x20, #5, MUL VL]\n"
      "zip1 z11.d, z2.d, z17.d\n"
      "zip2 z17.d, z2.d, z17.d\n"
      "zip1 z12.d, z1.d, z18.d\n"
      "zip2 z18.d, z1.d, z18.d\n"
      "zip1 z13.d, z20.d, z19.d\n"
      "zip2 z19.d, z20.d, z19.d\n"
      "zip1 z20.d, z21.d, z26.d\n"
      "zip2 z26.d, z21.d, z26.d\n"
      "zip1 z21.d, z22.d, z27.d\n"
      "zip2 z27.d, z22.d, z27.d\n"
      "zip1 z22.d, z23.d, z28.d\n"
      "zip2 z28.d, z23.d, z28.d\n"
      "zip1 z23.d, z24.d, z29.d\n"
      "zip2 z29.d, z24.d, z29.d\n"
      "zip1 z24.d, z25.d, z30.d\n"
      "zip2 z30.d, z25.d, z30.d\n"
      "zip1 z25.d, z0.d, z31.d\n"
      "zip2 z31.d, z0.d, z31.d\n"
      "b 34f\n"
      "33:"  // Height 3: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "mov z20.b, #0x0\n"
      "mov z21.b, #0x0\n"
      "mov z22.b, #0x0\n"
      "mov z23.b, #0x0\n"
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "mov z27.b, #0x0\n"
      "mov z28.b, #0x0\n"
      "mov z29.b, #0x0\n"
      "mov z30.b, #0x0\n"
      "mov z31.b, #0x0\n"
      "34:"  // Height 3: setup done
      "mov x26, #0x0\n"
      "35:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 36f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "cbnz x26, 37f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "b 37f\n"
      "36:"  // Height 3: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "37:"  // Height 3: input setup done
      "cmp x25, #0x4\n"
      "ble 39f\n"
      "38:"  // Height 3: Multiply loop: Main loop head
      "whilelt p0.s, XZR, x25\n"
      "ld1h { z3.h }, p7/Z, [x12]\n"
      "ld1h { z2.h }, p7/Z, [x12, #1, MUL VL]\n"
      "sub x25, x25, #0x4\n"
      "ld1h { z1.h }, p7/Z, [x11]\n"
      "ld1h { z6.h }, p7/Z, [x11, #1, MUL VL]\n"
      "cmp x25, #0x4\n"
      "addvl x12, x12, #2\n"
      "addvl x11, x11, #2\n"
      "ld1rqw { z5.s }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "ld1rqw { z0.s }, p0/Z, [x23]\n"
      "ld1rqw { z4.s }, p0/Z, [x22]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      ".inst 0x658abca5  // bfcvt z5.h, p7/M, z5.s\n"
      ".inst 0x658abc00  // bfcvt z0.h, p7/M, z0.s\n"
      "uzp1 z5.h, z5.h, z5.h\n"
      ".inst 0x658abc84  // bfcvt z4.h, p7/M, z4.s\n"
      "uzp1 z0.h, z0.h, z0.h\n"
      "trn1 z5.d, z5.d, z0.d\n"
      "uzp1 z4.h, z4.h, z4.h\n"
      ".inst 0x6463e4a8  // bfmmla z8.s, z5.h, z3.h\n"
      ".inst 0x6463e494  // bfmmla z20.s, z4.h, z3.h\n"
      "ld1h { z3.h }, p7/Z, [x10]\n"
      ".inst 0x6462e4ae  // bfmmla z14.s, z5.h, z2.h\n"
      ".inst 0x6462e49a  // bfmmla z26.s, z4.h, z2.h\n"
      "ld1h { z2.h }, p7/Z, [x10, #1, MUL VL]\n"
      "addvl x10, x10, #2\n"
      ".inst 0x6461e4a9  // bfmmla z9.s, z5.h, z1.h\n"
      ".inst 0x6461e495  // bfmmla z21.s, z4.h, z1.h\n"
      ".inst 0x6466e4af  // bfmmla z15.s, z5.h, z6.h\n"
      "ld1h { z1.h }, p7/Z, [x9]\n"
      ".inst 0x6466e49b  // bfmmla z27.s, z4.h, z6.h\n"
      "ld1h { z0.h }, p7/Z, [x9, #1, MUL VL]\n"
      ".inst 0x6463e4aa  // bfmmla z10.s, z5.h, z3.h\n"
      "addvl x9, x9, #2\n"
      ".inst 0x6463e496  // bfmmla z22.s, z4.h, z3.h\n"
      ".inst 0x6462e4b0  // bfmmla z16.s, z5.h, z2.h\n"
      "ld1h { z3.h }, p7/Z, [x28]\n"
      ".inst 0x6462e49c  // bfmmla z28.s, z4.h, z2.h\n"
      ".inst 0x6461e4ab  // bfmmla z11.s, z5.h, z1.h\n"
      "ld1h { z2.h }, p7/Z, [x28, #1, MUL VL]\n"
      "addvl x28, x28, #2\n"
      ".inst 0x6461e497  // bfmmla z23.s, z4.h, z1.h\n"
      ".inst 0x6460e4b1  // bfmmla z17.s, z5.h, z0.h\n"
      "ld1h { z1.h }, p7/Z, [x27]\n"
      ".inst 0x6460e49d  // bfmmla z29.s, z4.h, z0.h\n"
      "ld1h { z0.h }, p7/Z, [x27, #1, MUL VL]\n"
      ".inst 0x6463e4ac  // bfmmla z12.s, z5.h, z3.h\n"
      "addvl x27, x27, #2\n"
      ".inst 0x6463e498  // bfmmla z24.s, z4.h, z3.h\n"
      ".inst 0x6462e4b2  // bfmmla z18.s, z5.h, z2.h\n"
      ".inst 0x6462e49e  // bfmmla z30.s, z4.h, z2.h\n"
      ".inst 0x6461e4ad  // bfmmla z13.s, z5.h, z1.h\n"
      ".inst 0x6461e499  // bfmmla z25.s, z4.h, z1.h\n"
      ".inst 0x6460e4b3  // bfmmla z19.s, z5.h, z0.h\n"
      ".inst 0x6460e49f  // bfmmla z31.s, z4.h, z0.h\n"
      "bgt 38b\n"
      "39:"  // Height 3: Multiply loop: Single iteration only
      "whilelt p0.s, XZR, x25\n"
      "ld1h { z3.h }, p7/Z, [x12]\n"
      "ld1h { z2.h }, p7/Z, [x12, #1, MUL VL]\n"
      "addvl x12, x12, #2\n"
      "ld1h { z6.h }, p7/Z, [x11]\n"
      "ld1h { z1.h }, p7/Z, [x11, #1, MUL VL]\n"
      "addvl x11, x11, #2\n"
      "ld1rqw { z5.s }, p0/Z, [x24]\n"
      "ld1rqw { z0.s }, p0/Z, [x23]\n"
      "ld1rqw { z4.s }, p0/Z, [x22]\n"
      ".inst 0x658abca5  // bfcvt z5.h, p7/M, z5.s\n"
      ".inst 0x658abc00  // bfcvt z0.h, p7/M, z0.s\n"
      "uzp1 z5.h, z5.h, z5.h\n"
      "uzp1 z0.h, z0.h, z0.h\n"
      ".inst 0x658abc84  // bfcvt z4.h, p7/M, z4.s\n"
      "trn1 z5.d, z5.d, z0.d\n"
      "uzp1 z4.h, z4.h, z4.h\n"
      ".inst 0x6463e4a8  // bfmmla z8.s, z5.h, z3.h\n"
      ".inst 0x6462e4ae  // bfmmla z14.s, z5.h, z2.h\n"
      ".inst 0x6466e4a9  // bfmmla z9.s, z5.h, z6.h\n"
      ".inst 0x6461e4af  // bfmmla z15.s, z5.h, z1.h\n"
      ".inst 0x6463e494  // bfmmla z20.s, z4.h, z3.h\n"
      "ld1h { z3.h }, p7/Z, [x10]\n"
      ".inst 0x6462e49a  // bfmmla z26.s, z4.h, z2.h\n"
      "ld1h { z2.h }, p7/Z, [x10, #1, MUL VL]\n"
      ".inst 0x6466e495  // bfmmla z21.s, z4.h, z6.h\n"
      ".inst 0x6461e49b  // bfmmla z27.s, z4.h, z1.h\n"
      "ld1h { z1.h }, p7/Z, [x9]\n"
      "ld1h { z0.h }, p7/Z, [x9, #1, MUL VL]\n"
      "addvl x10, x10, #2\n"
      "addvl x9, x9, #2\n"
      ".inst 0x6463e4aa  // bfmmla z10.s, z5.h, z3.h\n"
      ".inst 0x6463e496  // bfmmla z22.s, z4.h, z3.h\n"
      ".inst 0x6462e4b0  // bfmmla z16.s, z5.h, z2.h\n"
      ".inst 0x6462e49c  // bfmmla z28.s, z4.h, z2.h\n"
      "ld1h { z3.h }, p7/Z, [x28]\n"
      "ld1h { z2.h }, p7/Z, [x28, #1, MUL VL]\n"
      ".inst 0x6461e4ab  // bfmmla z11.s, z5.h, z1.h\n"
      ".inst 0x6461e497  // bfmmla z23.s, z4.h, z1.h\n"
      "ld1h { z1.h }, p7/Z, [x27]\n"
      "addvl x28, x28, #2\n"
      ".inst 0x6460e4b1  // bfmmla z17.s, z5.h, z0.h\n"
      ".inst 0x6460e49d  // bfmmla z29.s, z4.h, z0.h\n"
      "ld1h { z0.h }, p7/Z, [x27, #1, MUL VL]\n"
      "addvl x27, x27, #2\n"
      ".inst 0x6463e4ac  // bfmmla z12.s, z5.h, z3.h\n"
      ".inst 0x6463e498  // bfmmla z24.s, z4.h, z3.h\n"
      ".inst 0x6462e4b2  // bfmmla z18.s, z5.h, z2.h\n"
      ".inst 0x6462e49e  // bfmmla z30.s, z4.h, z2.h\n"
      ".inst 0x6461e4ad  // bfmmla z13.s, z5.h, z1.h\n"
      ".inst 0x6461e499  // bfmmla z25.s, z4.h, z1.h\n"
      ".inst 0x6460e4b3  // bfmmla z19.s, z5.h, z0.h\n"
      ".inst 0x6460e49f  // bfmmla z31.s, z4.h, z0.h\n"
      "40:"  // Height 3: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 35b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 z4.d, z8.d, z14.d\n"
      "uzp2 z8.d, z8.d, z14.d\n"
      "uzp1 z14.d, z9.d, z15.d\n"
      "uzp2 z9.d, z9.d, z15.d\n"
      "uzp1 z15.d, z10.d, z16.d\n"
      "uzp2 z10.d, z10.d, z16.d\n"
      "add x24, x13, x20, LSL #2\n"
      "uzp1 z16.d, z11.d, z17.d\n"
      "uzp2 z11.d, z11.d, z17.d\n"
      "add x23, x24, x20, LSL #2\n"
      "uzp1 z17.d, z12.d, z18.d\n"
      "uzp2 z12.d, z12.d, z18.d\n"
      "uzp1 z18.d, z13.d, z19.d\n"
      "uzp2 z13.d, z13.d, z19.d\n"
      "uzp1 z20.d, z20.d, z26.d\n"
      "uzp1 z21.d, z21.d, z27.d\n"
      "uzp1 z22.d, z22.d, z28.d\n"
      "uzp1 z23.d, z23.d, z29.d\n"
      "uzp1 z24.d, z24.d, z30.d\n"
      "uzp1 z25.d, z25.d, z31.d\n"
      "tbz %x[flags], #1, 41f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z0.s }, p7/Z, [x21]\n"
      "ld1rw { z19.s }, p7/Z, [x20]\n"
      "fmin z4.s, p7/M, z4.s, z0.s\n"
      "fmin z14.s, p7/M, z14.s, z0.s\n"
      "fmin z15.s, p7/M, z15.s, z0.s\n"
      "fmin z16.s, p7/M, z16.s, z0.s\n"
      "fmin z17.s, p7/M, z17.s, z0.s\n"
      "fmin z18.s, p7/M, z18.s, z0.s\n"
      "fmin z8.s, p7/M, z8.s, z0.s\n"
      "fmin z9.s, p7/M, z9.s, z0.s\n"
      "fmin z10.s, p7/M, z10.s, z0.s\n"
      "fmin z11.s, p7/M, z11.s, z0.s\n"
      "fmin z12.s, p7/M, z12.s, z0.s\n"
      "fmin z13.s, p7/M, z13.s, z0.s\n"
      "fmin z20.s, p7/M, z20.s, z0.s\n"
      "fmin z21.s, p7/M, z21.s, z0.s\n"
      "fmin z22.s, p7/M, z22.s, z0.s\n"
      "fmin z23.s, p7/M, z23.s, z0.s\n"
      "fmin z24.s, p7/M, z24.s, z0.s\n"
      "fmin z25.s, p7/M, z25.s, z0.s\n"
      "fmax z4.s, p7/M, z4.s, z19.s\n"
      "fmax z14.s, p7/M, z14.s, z19.s\n"
      "fmax z15.s, p7/M, z15.s, z19.s\n"
      "fmax z16.s, p7/M, z16.s, z19.s\n"
      "fmax z17.s, p7/M, z17.s, z19.s\n"
      "fmax z18.s, p7/M, z18.s, z19.s\n"
      "fmax z8.s, p7/M, z8.s, z19.s\n"
      "fmax z9.s, p7/M, z9.s, z19.s\n"
      "fmax z10.s, p7/M, z10.s, z19.s\n"
      "fmax z11.s, p7/M, z11.s, z19.s\n"
      "fmax z12.s, p7/M, z12.s, z19.s\n"
      "fmax z13.s, p7/M, z13.s, z19.s\n"
      "fmax z20.s, p7/M, z20.s, z19.s\n"
      "fmax z21.s, p7/M, z21.s, z19.s\n"
      "fmax z22.s, p7/M, z22.s, z19.s\n"
      "fmax z23.s, p7/M, z23.s, z19.s\n"
      "fmax z24.s, p7/M, z24.s, z19.s\n"
      "fmax z25.s, p7/M, z25.s, z19.s\n"
      "41:"  // Height 3: No activation
      "st1w { z4.s }, p6, [x13]\n"
      "st1w { z14.s }, p5, [x13, #1, MUL VL]\n"
      "st1w { z15.s }, p4, [x13, #2, MUL VL]\n"
      "st1w { z16.s }, p3, [x13, #3, MUL VL]\n"
      "st1w { z17.s }, p2, [x13, #4, MUL VL]\n"
      "st1w { z18.s }, p1, [x13, #5, MUL VL]\n"
      "addvl x13, x13, #6\n"
      "st1w { z8.s }, p6, [x24]\n"
      "st1w { z9.s }, p5, [x24, #1, MUL VL]\n"
      "st1w { z10.s }, p4, [x24, #2, MUL VL]\n"
      "st1w { z11.s }, p3, [x24, #3, MUL VL]\n"
      "st1w { z12.s }, p2, [x24, #4, MUL VL]\n"
      "st1w { z13.s }, p1, [x24, #5, MUL VL]\n"
      "st1w { z20.s }, p6, [x23]\n"
      "st1w { z21.s }, p5, [x23, #1, MUL VL]\n"
      "st1w { z22.s }, p4, [x23, #2, MUL VL]\n"
      "st1w { z23.s }, p3, [x23, #3, MUL VL]\n"
      "st1w { z24.s }, p2, [x23, #4, MUL VL]\n"
      "st1w { z25.s }, p1, [x23, #5, MUL VL]\n"
      "42:"  // Height 3: Writeback done
      "decw x14, ALL, MUL #6\n"
      "cmp x14, XZR\n"
      "bgt 30b\n"
      "b 58f\n"
      "43:"  // Height 4
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "mov x21, #0x10\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "madd x21, x20, x21, x13\n"
      "str x21, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "44:"  // Height 4: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cntw x21, ALL, MUL #5\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x28, x9, x20, LSL #1\n"
      "add x27, x28, x20, LSL #1\n"
      "add x20, x27, x20, LSL #1\n"
      "cmp x14, x21\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 45f\n"
      "decw x21\n"
      "mov x27, x12\n"
      "cmp x14, x21\n"
      "bgt 45f\n"
      "decw x21\n"
      "mov x28, x12\n"
      "cmp x14, x21\n"
      "bgt 45f\n"
      "decw x21\n"
      "mov x9, x12\n"
      "cmp x14, x21\n"
      "bgt 45f\n"
      "decw x21\n"
      "mov x10, x12\n"
      "cmp x14, x21\n"
      "bgt 45f\n"
      "mov x11, x12\n"
      "45:"  // Height 4: B setup done
      "mov x20, #0x0\n"
      "whilelt p6.s, x20, x14\n"
      "incw x20\n"
      "whilelt p5.s, x20, x14\n"
      "incw x20\n"
      "whilelt p4.s, x20, x14\n"
      "incw x20\n"
      "whilelt p3.s, x20, x14\n"
      "incw x20\n"
      "whilelt p2.s, x20, x14\n"
      "incw x20\n"
      "whilelt p1.s, x20, x14\n"
      "cbz x15, 46f\n"
      "ld1w { z8.s }, p7/Z, [x15]\n"
      "ld1w { z9.s }, p7/Z, [x15, #1, MUL VL]\n"
      "ld1w { z10.s }, p7/Z, [x15, #2, MUL VL]\n"
      "ld1w { z11.s }, p7/Z, [x15, #3, MUL VL]\n"
      "ld1w { z12.s }, p7/Z, [x15, #4, MUL VL]\n"
      "ld1w { z13.s }, p7/Z, [x15, #5, MUL VL]\n"
      "addvl x15, x15, #6\n"
      "zip2 z14.d, z8.d, z8.d\n"
      "zip1 z8.d, z8.d, z8.d\n"
      "zip2 z15.d, z9.d, z9.d\n"
      "zip1 z9.d, z9.d, z9.d\n"
      "zip2 z16.d, z10.d, z10.d\n"
      "zip1 z10.d, z10.d, z10.d\n"
      "zip2 z17.d, z11.d, z11.d\n"
      "zip1 z11.d, z11.d, z11.d\n"
      "zip2 z18.d, z12.d, z12.d\n"
      "zip1 z12.d, z12.d, z12.d\n"
      "zip2 z19.d, z13.d, z13.d\n"
      "zip1 z13.d, z13.d, z13.d\n"
      "mov z20.d, z8.d\n"
      "mov z26.d, z14.d\n"
      "mov z21.d, z9.d\n"
      "mov z27.d, z15.d\n"
      "mov z22.d, z10.d\n"
      "mov z28.d, z16.d\n"
      "mov z23.d, z11.d\n"
      "mov z29.d, z17.d\n"
      "mov z24.d, z12.d\n"
      "mov z30.d, z18.d\n"
      "mov z25.d, z13.d\n"
      "mov z31.d, z19.d\n"
      "b 48f\n"
      "46:"  // Height 4: no bias
      "tbz %x[flags], #0, 47f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z22.s }, p6/Z, [x13]\n"
      "ld1w { z24.s }, p5/Z, [x13, #1, MUL VL]\n"
      "ld1w { z26.s }, p4/Z, [x13, #2, MUL VL]\n"
      "ld1w { z27.s }, p3/Z, [x13, #3, MUL VL]\n"
      "ld1w { z29.s }, p2/Z, [x13, #4, MUL VL]\n"
      "add x22, x13, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z20.s }, p1/Z, [x13, #5, MUL VL]\n"
      "ld1w { z14.s }, p6/Z, [x22]\n"
      "ld1w { z15.s }, p5/Z, [x22, #1, MUL VL]\n"
      "ld1w { z16.s }, p4/Z, [x22, #2, MUL VL]\n"
      "ld1w { z17.s }, p3/Z, [x22, #3, MUL VL]\n"
      "ld1w { z18.s }, p2/Z, [x22, #4, MUL VL]\n"
      "ld1w { z19.s }, p1/Z, [x22, #5, MUL VL]\n"
      "ld1w { z21.s }, p6/Z, [x21]\n"
      "zip1 z8.d, z22.d, z14.d\n"
      "zip2 z14.d, z22.d, z14.d\n"
      "ld1w { z22.s }, p5/Z, [x21, #1, MUL VL]\n"
      "ld1w { z23.s }, p4/Z, [x21, #2, MUL VL]\n"
      "zip1 z9.d, z24.d, z15.d\n"
      "zip2 z15.d, z24.d, z15.d\n"
      "ld1w { z24.s }, p3/Z, [x21, #3, MUL VL]\n"
      "ld1w { z25.s }, p2/Z, [x21, #4, MUL VL]\n"
      "zip1 z10.d, z26.d, z16.d\n"
      "zip2 z16.d, z26.d, z16.d\n"
      "ld1w { z0.s }, p1/Z, [x21, #5, MUL VL]\n"
      "ld1w { z26.s }, p6/Z, [x20]\n"
      "zip1 z11.d, z27.d, z17.d\n"
      "zip2 z17.d, z27.d, z17.d\n"
      "ld1w { z27.s }, p5/Z, [x20, #1, MUL VL]\n"
      "ld1w { z28.s }, p4/Z, [x20, #2, MUL VL]\n"
      "zip1 z12.d, z29.d, z18.d\n"
      "zip2 z18.d, z29.d, z18.d\n"
      "ld1w { z29.s }, p3/Z, [x20, #3, MUL VL]\n"
      "ld1w { z30.s }, p2/Z, [x20, #4, MUL VL]\n"
      "zip1 z13.d, z20.d, z19.d\n"
      "zip2 z19.d, z20.d, z19.d\n"
      "ld1w { z31.s }, p1/Z, [x20, #5, MUL VL]\n"
      "zip1 z20.d, z21.d, z26.d\n"
      "zip2 z26.d, z21.d, z26.d\n"
      "zip1 z21.d, z22.d, z27.d\n"
      "zip2 z27.d, z22.d, z27.d\n"
      "zip1 z22.d, z23.d, z28.d\n"
      "zip2 z28.d, z23.d, z28.d\n"
      "zip1 z23.d, z24.d, z29.d\n"
      "zip2 z29.d, z24.d, z29.d\n"
      "zip1 z24.d, z25.d, z30.d\n"
      "zip2 z30.d, z25.d, z30.d\n"
      "zip1 z25.d, z0.d, z31.d\n"
      "zip2 z31.d, z0.d, z31.d\n"
      "b 48f\n"
      "47:"  // Height 4: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "mov z20.b, #0x0\n"
      "mov z21.b, #0x0\n"
      "mov z22.b, #0x0\n"
      "mov z23.b, #0x0\n"
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "mov z27.b, #0x0\n"
      "mov z28.b, #0x0\n"
      "mov z29.b, #0x0\n"
      "mov z30.b, #0x0\n"
      "mov z31.b, #0x0\n"
      "48:"  // Height 4: setup done
      "mov x26, #0x0\n"
      "49:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 50f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "ldr x21, [x20, #0x18]\n"
      "cbnz x26, 51f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "add x21, x21, x20, LSL #2\n"
      "b 51f\n"
      "50:"  // Height 4: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "add x21, x22, x21, LSL #2\n"
      "51:"  // Height 4: input setup done
      "cmp x25, #0x4\n"
      "ble 53f\n"
      "52:"  // Height 4: Multiply loop: Main loop head
      "whilelt p0.s, XZR, x25\n"
      "ld1h { z3.h }, p7/Z, [x12]\n"
      "ld1h { z2.h }, p7/Z, [x12, #1, MUL VL]\n"
      "sub x25, x25, #0x4\n"
      "ld1h { z7.h }, p7/Z, [x11]\n"
      "ld1h { z6.h }, p7/Z, [x11, #1, MUL VL]\n"
      "cmp x25, #0x4\n"
      "addvl x12, x12, #2\n"
      "addvl x11, x11, #2\n"
      "ld1rqw { z5.s }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "ld1rqw { z1.s }, p0/Z, [x23]\n"
      "ld1rqw { z4.s }, p0/Z, [x22]\n"
      "ld1rqw { z0.s }, p0/Z, [x21]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      ".inst 0x658abca5  // bfcvt z5.h, p7/M, z5.s\n"
      ".inst 0x658abc21  // bfcvt z1.h, p7/M, z1.s\n"
      ".inst 0x658abc84  // bfcvt z4.h, p7/M, z4.s\n"
      ".inst 0x658abc00  // bfcvt z0.h, p7/M, z0.s\n"
      "uzp1 z5.h, z5.h, z5.h\n"
      "uzp1 z1.h, z1.h, z1.h\n"
      "uzp1 z4.h, z4.h, z4.h\n"
      "uzp1 z0.h, z0.h, z0.h\n"
      "trn1 z5.d, z5.d, z1.d\n"
      "trn1 z4.d, z4.d, z0.d\n"
      ".inst 0x6463e4a8  // bfmmla z8.s, z5.h, z3.h\n"
      ".inst 0x6463e494  // bfmmla z20.s, z4.h, z3.h\n"
      "ld1h { z3.h }, p7/Z, [x10]\n"
      ".inst 0x6462e4ae  // bfmmla z14.s, z5.h, z2.h\n"
      ".inst 0x6462e49a  // bfmmla z26.s, z4.h, z2.h\n"
      "ld1h { z2.h }, p7/Z, [x10, #1, MUL VL]\n"
      "addvl x10, x10, #2\n"
      ".inst 0x6467e4a9  // bfmmla z9.s, z5.h, z7.h\n"
      ".inst 0x6467e495  // bfmmla z21.s, z4.h, z7.h\n"
      ".inst 0x6466e4af  // bfmmla z15.s, z5.h, z6.h\n"
      "ld1h { z1.h }, p7/Z, [x9]\n"
      ".inst 0x6466e49b  // bfmmla z27.s, z4.h, z6.h\n"
      "ld1h { z0.h }, p7/Z, [x9, #1, MUL VL]\n"
      ".inst 0x6463e4aa  // bfmmla z10.s, z5.h, z3.h\n"
      "addvl x9, x9, #2\n"
      ".inst 0x6463e496  // bfmmla z22.s, z4.h, z3.h\n"
      ".inst 0x6462e4b0  // bfmmla z16.s, z5.h, z2.h\n"
      "ld1h { z3.h }, p7/Z, [x28]\n"
      ".inst 0x6462e49c  // bfmmla z28.s, z4.h, z2.h\n"
      ".inst 0x6461e4ab  // bfmmla z11.s, z5.h, z1.h\n"
      "ld1h { z2.h }, p7/Z, [x28, #1, MUL VL]\n"
      "addvl x28, x28, #2\n"
      ".inst 0x6461e497  // bfmmla z23.s, z4.h, z1.h\n"
      ".inst 0x6460e4b1  // bfmmla z17.s, z5.h, z0.h\n"
      "ld1h { z1.h }, p7/Z, [x27]\n"
      ".inst 0x6460e49d  // bfmmla z29.s, z4.h, z0.h\n"
      "ld1h { z0.h }, p7/Z, [x27, #1, MUL VL]\n"
      ".inst 0x6463e4ac  // bfmmla z12.s, z5.h, z3.h\n"
      "addvl x27, x27, #2\n"
      ".inst 0x6463e498  // bfmmla z24.s, z4.h, z3.h\n"
      ".inst 0x6462e4b2  // bfmmla z18.s, z5.h, z2.h\n"
      ".inst 0x6462e49e  // bfmmla z30.s, z4.h, z2.h\n"
      ".inst 0x6461e4ad  // bfmmla z13.s, z5.h, z1.h\n"
      ".inst 0x6461e499  // bfmmla z25.s, z4.h, z1.h\n"
      ".inst 0x6460e4b3  // bfmmla z19.s, z5.h, z0.h\n"
      ".inst 0x6460e49f  // bfmmla z31.s, z4.h, z0.h\n"
      "bgt 52b\n"
      "53:"  // Height 4: Multiply loop: Single iteration only
      "whilelt p0.s, XZR, x25\n"
      "ld1h { z3.h }, p7/Z, [x12]\n"
      "ld1h { z2.h }, p7/Z, [x12, #1, MUL VL]\n"
      "addvl x12, x12, #2\n"
      "ld1h { z7.h }, p7/Z, [x11]\n"
      "ld1h { z6.h }, p7/Z, [x11, #1, MUL VL]\n"
      "addvl x11, x11, #2\n"
      "ld1rqw { z5.s }, p0/Z, [x24]\n"
      "ld1rqw { z1.s }, p0/Z, [x23]\n"
      "ld1rqw { z4.s }, p0/Z, [x22]\n"
      "ld1rqw { z0.s }, p0/Z, [x21]\n"
      ".inst 0x658abca5  // bfcvt z5.h, p7/M, z5.s\n"
      ".inst 0x658abc21  // bfcvt z1.h, p7/M, z1.s\n"
      ".inst 0x658abc84  // bfcvt z4.h, p7/M, z4.s\n"
      ".inst 0x658abc00  // bfcvt z0.h, p7/M, z0.s\n"
      "uzp1 z5.h, z5.h, z5.h\n"
      "uzp1 z1.h, z1.h, z1.h\n"
      "uzp1 z4.h, z4.h, z4.h\n"
      "uzp1 z0.h, z0.h, z0.h\n"
      "trn1 z5.d, z5.d, z1.d\n"
      "trn1 z4.d, z4.d, z0.d\n"
      ".inst 0x6463e4a8  // bfmmla z8.s, z5.h, z3.h\n"
      ".inst 0x6463e494  // bfmmla z20.s, z4.h, z3.h\n"
      "ld1h { z3.h }, p7/Z, [x10]\n"
      ".inst 0x6462e4ae  // bfmmla z14.s, z5.h, z2.h\n"
      ".inst 0x6462e49a  // bfmmla z26.s, z4.h, z2.h\n"
      "ld1h { z2.h }, p7/Z, [x10, #1, MUL VL]\n"
      "addvl x10, x10, #2\n"
      ".inst 0x6467e4a9  // bfmmla z9.s, z5.h, z7.h\n"
      ".inst 0x6467e495  // bfmmla z21.s, z4.h, z7.h\n"
      ".inst 0x6466e4af  // bfmmla z15.s, z5.h, z6.h\n"
      "ld1h { z1.h }, p7/Z, [x9]\n"
      ".inst 0x6466e49b  // bfmmla z27.s, z4.h, z6.h\n"
      "ld1h { z0.h }, p7/Z, [x9, #1, MUL VL]\n"
      ".inst 0x6463e4aa  // bfmmla z10.s, z5.h, z3.h\n"
      "addvl x9, x9, #2\n"
      ".inst 0x6463e496  // bfmmla z22.s, z4.h, z3.h\n"
      ".inst 0x6462e4b0  // bfmmla z16.s, z5.h, z2.h\n"
      "ld1h { z3.h }, p7/Z, [x28]\n"
      ".inst 0x6462e49c  // bfmmla z28.s, z4.h, z2.h\n"
      ".inst 0x6461e4ab  // bfmmla z11.s, z5.h, z1.h\n"
      "ld1h { z2.h }, p7/Z, [x28, #1, MUL VL]\n"
      "addvl x28, x28, #2\n"
      ".inst 0x6461e497  // bfmmla z23.s, z4.h, z1.h\n"
      ".inst 0x6460e4b1  // bfmmla z17.s, z5.h, z0.h\n"
      "ld1h { z1.h }, p7/Z, [x27]\n"
      ".inst 0x6460e49d  // bfmmla z29.s, z4.h, z0.h\n"
      "ld1h { z0.h }, p7/Z, [x27, #1, MUL VL]\n"
      ".inst 0x6463e4ac  // bfmmla z12.s, z5.h, z3.h\n"
      "addvl x27, x27, #2\n"
      ".inst 0x6463e498  // bfmmla z24.s, z4.h, z3.h\n"
      ".inst 0x6462e4b2  // bfmmla z18.s, z5.h, z2.h\n"
      ".inst 0x6462e49e  // bfmmla z30.s, z4.h, z2.h\n"
      ".inst 0x6461e4ad  // bfmmla z13.s, z5.h, z1.h\n"
      ".inst 0x6461e499  // bfmmla z25.s, z4.h, z1.h\n"
      ".inst 0x6460e4b3  // bfmmla z19.s, z5.h, z0.h\n"
      ".inst 0x6460e49f  // bfmmla z31.s, z4.h, z0.h\n"
      "54:"  // Height 4: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 49b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 z4.d, z8.d, z14.d\n"
      "uzp2 z8.d, z8.d, z14.d\n"
      "uzp1 z14.d, z9.d, z15.d\n"
      "uzp2 z9.d, z9.d, z15.d\n"
      "uzp1 z15.d, z10.d, z16.d\n"
      "uzp2 z10.d, z10.d, z16.d\n"
      "add x24, x13, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "uzp1 z16.d, z11.d, z17.d\n"
      "uzp2 z11.d, z11.d, z17.d\n"
      "add x22, x23, x20, LSL #2\n"
      "uzp1 z17.d, z12.d, z18.d\n"
      "uzp2 z12.d, z12.d, z18.d\n"
      "uzp1 z18.d, z13.d, z19.d\n"
      "uzp2 z13.d, z13.d, z19.d\n"
      "uzp1 z19.d, z20.d, z26.d\n"
      "uzp2 z20.d, z20.d, z26.d\n"
      "uzp1 z26.d, z21.d, z27.d\n"
      "uzp2 z21.d, z21.d, z27.d\n"
      "uzp1 z27.d, z22.d, z28.d\n"
      "uzp2 z22.d, z22.d, z28.d\n"
      "uzp1 z28.d, z23.d, z29.d\n"
      "uzp2 z23.d, z23.d, z29.d\n"
      "uzp1 z29.d, z24.d, z30.d\n"
      "uzp2 z24.d, z24.d, z30.d\n"
      "uzp1 z30.d, z25.d, z31.d\n"
      "uzp2 z25.d, z25.d, z31.d\n"
      "tbz %x[flags], #1, 55f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z1.s }, p7/Z, [x21]\n"
      "ld1rw { z0.s }, p7/Z, [x20]\n"
      "fmin z4.s, p7/M, z4.s, z1.s\n"
      "fmin z14.s, p7/M, z14.s, z1.s\n"
      "fmin z15.s, p7/M, z15.s, z1.s\n"
      "fmin z16.s, p7/M, z16.s, z1.s\n"
      "fmin z17.s, p7/M, z17.s, z1.s\n"
      "fmin z18.s, p7/M, z18.s, z1.s\n"
      "fmin z8.s, p7/M, z8.s, z1.s\n"
      "fmin z9.s, p7/M, z9.s, z1.s\n"
      "fmin z10.s, p7/M, z10.s, z1.s\n"
      "fmin z11.s, p7/M, z11.s, z1.s\n"
      "fmin z12.s, p7/M, z12.s, z1.s\n"
      "fmin z13.s, p7/M, z13.s, z1.s\n"
      "fmin z19.s, p7/M, z19.s, z1.s\n"
      "fmin z26.s, p7/M, z26.s, z1.s\n"
      "fmin z27.s, p7/M, z27.s, z1.s\n"
      "fmin z28.s, p7/M, z28.s, z1.s\n"
      "fmin z29.s, p7/M, z29.s, z1.s\n"
      "fmin z30.s, p7/M, z30.s, z1.s\n"
      "fmin z20.s, p7/M, z20.s, z1.s\n"
      "fmin z21.s, p7/M, z21.s, z1.s\n"
      "fmin z22.s, p7/M, z22.s, z1.s\n"
      "fmin z23.s, p7/M, z23.s, z1.s\n"
      "fmin z24.s, p7/M, z24.s, z1.s\n"
      "fmin z25.s, p7/M, z25.s, z1.s\n"
      "fmax z4.s, p7/M, z4.s, z0.s\n"
      "fmax z14.s, p7/M, z14.s, z0.s\n"
      "fmax z15.s, p7/M, z15.s, z0.s\n"
      "fmax z16.s, p7/M, z16.s, z0.s\n"
      "fmax z17.s, p7/M, z17.s, z0.s\n"
      "fmax z18.s, p7/M, z18.s, z0.s\n"
      "fmax z8.s, p7/M, z8.s, z0.s\n"
      "fmax z9.s, p7/M, z9.s, z0.s\n"
      "fmax z10.s, p7/M, z10.s, z0.s\n"
      "fmax z11.s, p7/M, z11.s, z0.s\n"
      "fmax z12.s, p7/M, z12.s, z0.s\n"
      "fmax z13.s, p7/M, z13.s, z0.s\n"
      "fmax z19.s, p7/M, z19.s, z0.s\n"
      "fmax z26.s, p7/M, z26.s, z0.s\n"
      "fmax z27.s, p7/M, z27.s, z0.s\n"
      "fmax z28.s, p7/M, z28.s, z0.s\n"
      "fmax z29.s, p7/M, z29.s, z0.s\n"
      "fmax z30.s, p7/M, z30.s, z0.s\n"
      "fmax z20.s, p7/M, z20.s, z0.s\n"
      "fmax z21.s, p7/M, z21.s, z0.s\n"
      "fmax z22.s, p7/M, z22.s, z0.s\n"
      "fmax z23.s, p7/M, z23.s, z0.s\n"
      "fmax z24.s, p7/M, z24.s, z0.s\n"
      "fmax z25.s, p7/M, z25.s, z0.s\n"
      "55:"  // Height 4: No activation
      "st1w { z4.s }, p6, [x13]\n"
      "st1w { z14.s }, p5, [x13, #1, MUL VL]\n"
      "st1w { z15.s }, p4, [x13, #2, MUL VL]\n"
      "st1w { z16.s }, p3, [x13, #3, MUL VL]\n"
      "st1w { z17.s }, p2, [x13, #4, MUL VL]\n"
      "st1w { z18.s }, p1, [x13, #5, MUL VL]\n"
      "addvl x13, x13, #6\n"
      "st1w { z8.s }, p6, [x24]\n"
      "st1w { z9.s }, p5, [x24, #1, MUL VL]\n"
      "st1w { z10.s }, p4, [x24, #2, MUL VL]\n"
      "st1w { z11.s }, p3, [x24, #3, MUL VL]\n"
      "st1w { z12.s }, p2, [x24, #4, MUL VL]\n"
      "st1w { z13.s }, p1, [x24, #5, MUL VL]\n"
      "st1w { z19.s }, p6, [x23]\n"
      "st1w { z26.s }, p5, [x23, #1, MUL VL]\n"
      "st1w { z27.s }, p4, [x23, #2, MUL VL]\n"
      "st1w { z28.s }, p3, [x23, #3, MUL VL]\n"
      "st1w { z29.s }, p2, [x23, #4, MUL VL]\n"
      "st1w { z30.s }, p1, [x23, #5, MUL VL]\n"
      "st1w { z20.s }, p6, [x22]\n"
      "st1w { z21.s }, p5, [x22, #1, MUL VL]\n"
      "st1w { z22.s }, p4, [x22, #2, MUL VL]\n"
      "st1w { z23.s }, p3, [x22, #3, MUL VL]\n"
      "st1w { z24.s }, p2, [x22, #4, MUL VL]\n"
      "st1w { z25.s }, p1, [x22, #5, MUL VL]\n"
      "56:"  // Height 4: Writeback done
      "decw x14, ALL, MUL #6\n"
      "cmp x14, XZR\n"
      "bgt 44b\n"
      "subs %x[M], %x[M], #0x4\n"
      "beq 58f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 57f\n"
      "add x21, x21, #0x4\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "57:"  // Update direct input
      "mov x20, #0x10\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "58:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offset_max] "I" (offsetof(KernelArgs, maxval)), [offset_min] "I" (offsetof(KernelArgs, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_B_stride] "I" (offsetof(KernelArgs, B_stride)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_bias] "I" (offsetof(KernelArgs, bias)), [offsetof_cur_B_ptr] "I" (offsetof(KernelArgs, cur_B_ptr)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm
#endif  // ARM_COMPUTE_ENABLE_SVE
