// Copyright 2015 MongoDB Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <iostream>

#include <bsoncxx/builder/basic/document.hpp>
#include <bsoncxx/builder/basic/kvp.hpp>
#include <bsoncxx/json.hpp>
#include <mongocxx/client.hpp>
#include <mongocxx/instance.hpp>
#include <mongocxx/pipeline.hpp>
#include <mongocxx/uri.hpp>

using bsoncxx::builder::basic::kvp;
using bsoncxx::builder::basic::make_document;

int main(int, char**) {
    // The mongocxx::instance constructor and destructor initialize and shut down the driver,
    // respectively. Therefore, a mongocxx::instance must be created before using the driver and
    // must remain alive for as long as the driver is in use.
    mongocxx::instance inst{};
    mongocxx::client conn{mongocxx::uri{}};

    auto db = conn["test"];

    // Group documents by field and calculate count.
    {
        // @begin: cpp-group-documents-by-a-field-and-calculate-count
        mongocxx::pipeline stages;

        stages.group(
            make_document(kvp("_id", "$borough"), kvp("count", make_document(kvp("$sum", 1)))));

        auto cursor = db["restaurants"].aggregate(stages);

        for (auto&& doc : cursor) {
            std::cout << bsoncxx::to_json(doc) << std::endl;
        }
        // @end: cpp-group-documents-by-a-field-and-calculate-count
    }

    // Filter and group documents.
    {
        // @begin: cpp-filter-and-group-documents
        mongocxx::pipeline stages;

        stages.match(make_document(kvp("borough", "queens"), kvp("cuisine", "Brazilian")))
            .group(make_document(kvp("_id", "$address.zipcode"),
                                 kvp("count", make_document(kvp("$sum", 1)))));

        auto cursor = db["restaurants"].aggregate(stages);

        for (auto&& doc : cursor) {
            std::cout << bsoncxx::to_json(doc) << std::endl;
        }
        // @end: cpp-filter-and-group-documents
    }
}
