/*	JDBC_Data_Port

PIRL CVS  ID: JDBC_Data_Port.java,v 2.7 2012/04/16 06:08:57 castalia Exp

Copyright (C) 2001-2008  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Database;

import PIRL.Configuration.Configuration;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.Statement;
import java.sql.SQLException;

import java.util.Vector;
import java.util.Iterator;
import java.util.ListIterator;

import java.lang.ClassNotFoundException;


/**	A <i>JDBC_Data_Port</i> manages access to any JDBC database.
<p>
	A JDBC_Data_Port provides a partial implementation of the
	<i>Data_Port</i> interface. Only the {@link #Open(Configuration)
	<code>Open</code>} method is left to be implemented by the subclass
	that finalizes the Data_Port implementation for the specific type of
	JDBC database.
<p>
	@author		Bradford Castalia, UA/PIRL
	@version	2.7
	@see		Data_Port
	@see		Configuration
*/
public abstract class JDBC_Data_Port
	implements Data_Port
{
/*	The ID and Port_Type variables should be set by the subclass
	that finalizes this abstract class. It's constructor should call
	the protected ID_Type method with its ID and Port_Type variables.
*/
private String
	ID			= "PIRL.Database.JDBC_Data_Port (2.7 2012/04/16 06:08:57)",
	Port_Type	= null;

//	The Database Configuration containing database server access information.
private Configuration
	The_Configuration			= null;

//	The JDBC Conncetion to the database server.
private Connection
	JDBC_Connection				= null;

//	SQL_Listener objects.
private Vector
	SQL_Listeners				= new Vector ();

/**	Delimiter between catalog, table and field components when more
	than one component is specified in a reference.
<p>
	The component delimiter is initialzed when the Data_Port is {@link
	#Open_Data_Port(String) open}ed by the server specific subclass using
	the getCatalogSeparator value obtained from the database metadata.
<p>
	The default is ".".
*/
protected String
	Component_Delimiter			= ".";

/**	Determines if database entity identifiers are case sensitive.
<p>
	A database server that does not use case sensitive entity identifiers
	will coerce the identifiers to lowercase. When these identifiers are
	returned from a query they will not match user specified identifiers
	that are in mixed case.
<p>
	Because some database systems on some operating systems are not case
	sensitive in handling entity identifiers (the names of catalogs,
	tables, and field names) it may be necessary to enforce case
	insensitivity when matching user specified names against identifiers
	returned from the database server.
<p>
	The case sensitivity of identifiers is initialzed when the Data_Port
	is {@link #Open_Data_Port(String) open}ed by the server specific
	subclass by using the {@link
	DatabaseMetaData#supportsMixedCaseIdentifiers()} value obtained from
	the database metadata.
<p>
	The default, until the Data_Port has set the value, is to assume that
	database identifiers are not case sensitive.
<p>
	@see	#Case_Sensitive_Identifiers()
	@see	Database#Matches(String, String)
*/
protected boolean
	Case_Sensitive_Identifiers	= false;

/**	Determines if the database server schema structure is to be treated
	as the Database class "catalog" structure.
<p>
	The meaning of the term "catalog" to the Database class is the
	database server structure that contains a collection of data tables.
	For some database servers the term "database" is used for this
	structure (e.g. MySQL), but the same JDBC catalog structure applies.
	However, for some database servers the corresponding structure is a
	"schema" (e.g. PostgreSQL) and the JDBC schema structure must be used
	to access it rather than using its catalog structure.
<p>
	This flag is expected to be set by database specific subclasses that
	complete the Data_Port implementation. It will be used by methods
	that need to make the distinction when using a JDBC method.
*/
protected boolean
	Treat_Schema_As_Catalog		= false;

private String
	Database_Catalog_Term		= "catalog",
	Database_Schema_Term		= "schema";

//	New-Line sequence.
private static final String
	NL							= Database.NL;


//	DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_ALL			= -1,
	DEBUG_CONFIGURE		= 1 << 0,
	DEBUG_OPEN			= 1 << 1,
	DEBUG_DESCRIPTION	= 1 << 2,
	DEBUG_QUERY			= 1 << 3,
	DEBUG_UPDATE		= 1 << 4,
	DEBUG_UTILITY		= 1 << 5,

	DEBUG				= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs an unopened JDBC_Data_Port.
*/
public JDBC_Data_Port () {}

/*==============================================================================
	Accessors
*/
/**	Tests if the Data_Port currently has an active Connection.
<p>
	@return	true if the Data_Port has access to a database; false
		otherwise.
*/
public boolean is_Open ()
{
if ((DEBUG & DEBUG_OPEN) != 0)
	System.out.println
		(">-< JDBC_Data_Port.is_Open: " + (JDBC_Connection != null));
//	Data_Port Implementation
return JDBC_Connection != null;
}

/**	Gets the JDBC Connection object for the Data_Port.
<p>
	@return	The JDBC Connection object for the Data_Port. This will be
		null if the Data_Port does not have an active Connection.
*/
public Connection Connection ()
{
//	Data_Port Implementation
return JDBC_Connection;
}

/**	Adds a SQL_Listener to the list of listeners to be notified of
	SQL statements before they are sent to the database server.
<p>
	@param	listener	A SQL_Listener to be added. If null, or the
		listener is already listed, nothing is done.
	@return	This Data_Port.
	@see	SQL_Listener
*/
public Data_Port Add_SQL_Listener
	(
	SQL_Listener	listener
	)
{
//	Data_Port Implementation
synchronized (SQL_Listeners)
	{
	if (listener != null &&
		! SQL_Listeners.contains (listener))
		SQL_Listeners.add (listener);
	}
return this;
}

/**	Removes a SQL_Listener from the list of listeners to be notified of
	SQL statements before they are sent to the database server.
<p>
	@param	listener	A SQL_Listener to be removed.
	@return	true if the listener was removed; false if the listener was
		not listed.
	@see	SQL_Listener
*/
public boolean Remove_SQL_Listener
	(
	SQL_Listener	listener
	)
{
//	Data_Port Implementation
synchronized (SQL_Listeners)
	{return SQL_Listeners.remove (listener);}
}


private void Report_SQL_Statement
	(
	String	SQL
	)
{
//	Data_Port Implementation
synchronized (SQL_Listeners)
	{
	Iterator
		listeners = SQL_Listeners.iterator ();
	while (listeners.hasNext ())
		((SQL_Listener)listeners.next ()).SQL_Statement (SQL);
	}
}

/**	Used by the finalizer subclass to register its class
	identification and specific type name.
<p>
	@param	ID	The class identification string. This is recommended
		to be of the form "Package_name.Class_name (Version Date)".
	@param	Type	The name of the specific type of Data_Port. This is
		expected to be the value of the <code>{@link Database#TYPE
		TYPE}</code> parameter from the Configuration.
*/
protected void ID_Type
	(
	String	ID,
	String	Type
	)
{
if (ID != null)
	this.ID += NL + ID;
if (Type == null)
	Port_Type = "";
else
	Port_Type = Type;
}

/*==============================================================================
	Methods
*/
/**	Gets the Configuration of the Data_Port.
<p>
	@return	The current Configuration of the Data_Port.
	@see	Configuration
*/
public Configuration Configuration ()
{return The_Configuration;}

/*------------------------------------------------------------------------------
	Access
*/
/**	Opens a connection to a database.
<p>
	This is a Data_Port interface method that is left to subclasses to
	implement. <b>Note</b>: This is the only abstract method.
<p>
	Subclasses are expected to implement the Open method as follows:
<ol
	start="1"
	type="1">
	<li>Update the supplied configuration with any local parameters.
		<p>
		This is typically done by using its <code>{@link
		Configuration#Set_Conditionally(String[][]) Set_Conditionally
		(Parameters[][])}<code> methods where Parameters[][] is an
		array of parameter name, value string pairs. There will a set
		of Required Parameters and a set of Optional Parameters.
		<b>Note</b>: The <code>{@link Database#DRIVER DRIVER}<code>
		parameter is required. Its value is the classname for the JDBC
		driver used with the specific type of Data_Port being
		implemented.
		<p>
	<li>Call the <code>{@link #Configure(Configuration)
		Configure}</code> method with the configuration.
		<p>
		This method will check that the Data_Port is not already open,
		confirm the presence of required parameters, and load the
		driver for the Data_Port.
		<p>
	<li>Assemble the URL string used by the JDBC driver.
		<p>
		This is driver specific and generally uses parameters from the
		configuration. The best way to get configuration parameter
		values is to use the protected <code>{@link
		#Config_Value(String) Config_Value}</code> method.
		<p>
	<li>Call the <code>{@link #Open_Data_Port(String)
		Open_Data_Port}</code> method with the URL.
		<p>
		This will invoke the JDBC DriverManager to make the connection.
</ol>
<p>
	@param	configuration	The Data_Port Configuration.
	@throws	Database_Exception	If the Data_Port could not be opened.
*/
public abstract void Open
	(
	Configuration	configuration
	)
	throws Database_Exception;

/**	Used by the finalizer subclass to register its Configuration and load
	its database driver.
<p>
	The Configuration must contain a <code>{@link Database#DRIVER
	DRIVER}</code> parameter. The value of this parameter is the class
	name of the driver. An instance of this class will be loaded for
	use.
<p>
	@param	configuration	The Configuration of the specific
		Data_Port. This is expected to be the Configuration passed to
		it via its <code>{@link #Open(Configuration) Open}</code>
		method, modified as appropriate.
	@throws	Database_Exception	If the Data_Port is already open, the
		Configuration does not contain the necessary
		<code>DRIVER</code> parameter, or the driver could not be
		loaded.
*/
protected void Configure
	(
	Configuration	configuration
	)
	throws Database_Exception
{
if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Configure:");
if (is_Open ())
	throw new Database_Exception
		(
		ID + NL
		+"The Data Port is already open."
		);

//	Setup the configuration.
if (configuration == null)
	throw new Database_Exception
		(
		ID + NL
		+"An invalid (null) configuration was specified."
		);
The_Configuration = configuration;

//	Load the driver.
String
	driver_name = The_Configuration.Get_Linked_One (Database.DRIVER);
if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println
		("    Driver: " + driver_name);
if (driver_name == null)
	throw new Database_Exception
		(
		ID + NL
		+"A \"" + Database.DRIVER + "\" parameter is required."
		);
try
	{
	/*
		Theorectically a new instance of the driver class is unnecessary.
		However, the documentation for the mm.mysql.jdbc driver notes
		that this should be done as a workaround for some "broken" Java
		implementations.

	Class.forName (driver_name).newInstance ();
	*/
	Class.forName (driver_name);
	}
catch (Exception exception)
	{
	throw new Database_Exception
		(
		ID + NL
		+"Unable to " +
			((exception instanceof ClassNotFoundException) ? "find" : "load") +
			" the JDBC driver: " + driver_name,
		exception
		);
	}
if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println
		("<<< JDBC_Data_Port.Configure:");
}

/**	Used by the finalizer subclass to get a JDBC Connection to the
	database as specified by its URL.
<p>
	@param	URL	The URL provided to the <code>{@link
		DriverManager#getConnection (String)
		DriverManager.getConnection} </code> method.  The syntax of the
		URL is dependent on the database driver being used. It is
		likely to contain specifications obtained from the Data_Port
		Configuration.
	@throws	Database_Exception	If the JDBC Connection failed.
*/
protected void Open_Data_Port
	(
	String	URL
	)
	throws Database_Exception
{
if ((DEBUG & DEBUG_OPEN) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Open_Data_Port: " + URL);
//	Open the data port connection.
try {JDBC_Connection = DriverManager.getConnection (URL);}
catch (SQLException exception)
	{
	//	Mask passwords in the SQL exception messages.
	throw new Database_Exception
		(
		Description () + NL
		+"Connection failed to: " + URL,
		Database_Exception.masked_SQLException (exception)
		);
	}

try
	{
	DatabaseMetaData
		metadata = JDBC_Connection.getMetaData ();

	//	Initialize identifier case sensitivity.
	Case_Sensitive_Identifiers =
		metadata.supportsMixedCaseIdentifiers ();

	//	Initialize the database server "catalog" term.
	Database_Catalog_Term = metadata.getCatalogTerm ();

	//	Initialize the database server "schema" term.
	Database_Schema_Term = metadata.getSchemaTerm ();

	//	Initialize the catalog and table delimiter.
	Component_Delimiter = metadata.getCatalogSeparator ();
	if ((DEBUG & DEBUG_OPEN) != 0)
		System.out.println
			("    Case_Sensitive_Identifiers = "
				+ Case_Sensitive_Identifiers + NL
			+"         Database_Catalog_Term = "
				+ Database_Catalog_Term + NL
			+"          Database_Schema_Term = "
				+ Database_Schema_Term + NL
			+"           Component_Delimiter = "
				+ Component_Delimiter);
	}
catch (Exception exception) {/* Leave the defaults */}
if ((DEBUG & DEBUG_OPEN) != 0)
	System.out.println
		("<<< JDBC_Data_Port.Open_Data_Port");
}

/**	Closes the JDBC Connection.
<p>
	If the Connection is already closed, nothing is done.
<p>
	@throws	Database_Exception	If there was a problem closing the JDBC
		Connection.
*/
public void Close ()
	throws Database_Exception
{
//	Data_Port Implementation
if ((DEBUG & DEBUG_OPEN) != 0)
	System.out.println
		(">-< JDBC_Data_Port.Close");
if (is_Open ())
	{
	String
		description = Description ();
	try {JDBC_Connection.close ();}
	catch (Exception exception)
		{
		throw new Database_Exception
			(
			description + NL
			+"Problem closing data port.",
			exception
			);
		}
	finally {JDBC_Connection = null;}
	}
}

/*------------------------------------------------------------------------------
	Description
*/
/**	Provides a String identifying the Data_Port.
<p>
	The identification includes the Data_Port Type, if known, the
	class ID for this JDBC_Data_Port and the class ID for the
	finalizer subclass.
<p>
	@return	The identifification for the Data_Port.
*/
public String toString ()
{
//	Data_Port Implementation
String
	string = "";
if (Port_Type != null &&
	Port_Type.length () > 0)
	string += Port_Type + ' ';
string += "Data Port:" + NL
		+ ID;
return string;
}

/**	Provides a multi-line description of the Data_Port.
<p>
	If the Data_Port is currently <code>{@link #Open Open}</code>, a
	description of the database connection may be included.
<p>
	@return The multi-line description of the Data_Port and any active
		database connection.
*/
public String Description ()
{
//	Data_Port Implementation
String
	description = toString ();
String
	string = Config_Value (Configuration.USER);
if (string.length () == 0)
	string = "(none)";
description += NL +
	"    User: " + string;
string = Config_Value (Configuration.HOST);
if (string.length () == 0)
	string = "(none)";
description += NL +
	"    Host: " + string;
if (is_Open ())
	{
	try
		{
		DatabaseMetaData
			metadata = JDBC_Connection.getMetaData ();
		description += NL +
			"  Driver: " + metadata.getDriverName () +
				" (" + Config_Value (Database.DRIVER) + ")" + NL +
			"          version " + metadata.getDriverVersion () + NL +
			"  Server: " + metadata.getDatabaseProductName () +
				" version " + metadata.getDatabaseProductVersion ();
		}
	catch (Exception exception) {}
	}
else
	description += NL +
		"Not connected to a database.";
return description;
}

/**	Gets a String describing the contents of selected catalogs and
	tables.
<p>
	For each table being described each field name and its data
	type is listed.
<p>
	@param	catalog	The catalog to have its contents described. If it is
		in {@link #Table_Reference(String, String) table reference}
		format, the catalog portion of the name will be used. If null,
		all catalogs on the database server will be described.
	@param	table	The table to have its contents described. If it is in
		{@link #Table_Reference(String, String) table reference} format,
		the table portion of the name will be used. If null, all tables
		in the catalog (or all catalogs) will be described.
	@return	A descriptive String.
*/
public String Contents
	(
	String	catalog,
	String	table
	)
{
//	Data_Port Implementation
String
	description = "";
if (is_Open ())
	{
	try
		{
        DatabaseMetaData
			metadata = JDBC_Connection.getMetaData ();
		Vector
			catalog_collection = Catalogs (),
			catalogs,
			table_collection,
			tables = null,
			names,
			types,
			sizes;

        if (catalog == null)
            catalogs = catalog_collection;
		else
			{
			String
				name = Catalog_Name (catalog);
			if (name.length () == 0)
				//	Take the argument String as-is;
				name = catalog;
			if (! Database.Matches
					(catalog_collection, name, Case_Sensitive_Identifiers))
				return
					"Unable to obtain the Contents of catalog \"" +
						name  + "\"." + NL +
					"It doesn't exist or access is denied.";
			catalogs = new Vector (1);
			catalogs.add (name);
			}

		if (table != null)
			{
			tables = new Vector (1);
			tables.add (table_name ("get Contents", table));
			}

		description +=
			"Identifiers are " + (Case_Sensitive_Identifiers ? "" : "not ")
				+ "case sensitive." + NL;
		if (Treat_Schema_As_Catalog)
			description +=
				"Database server schema are treated as Database catalogs." + NL;
		description += NL;

		//	Catalog descriptions:
        Iterator
			catalog_list = catalogs.iterator ();
        while (catalog_list.hasNext ())
            {
            catalog = (String)catalog_list.next ();
			table_collection = Tables (catalog);
			if (table == null)
				tables = table_collection;
			else if (! Database.Matches
						(table_collection, table, Case_Sensitive_Identifiers))
				continue;
			description +=
				catalog + " " + Database_Catalog_Term + ":" + NL;

			//	Table descriptions:
			Iterator
				table_list = tables.iterator ();
			while (table_list.hasNext ())
				{
				String
					catalog_table = (String)table_list.next ();
				description += "  " + catalog_table + " table" + NL;
				catalog_table = catalog + Component_Delimiter + catalog_table;

				//	Field descriptions:
				names = Field_Names (catalog_table);
				types = Field_Types (catalog_table);
				description +=
					"    Fields" + NL +
					"    ------" + NL;
				for (int field = 0;
						 field < names.size ();
						 field++)
					description +=
						"    " + field + " - " +
						names.elementAt (field) + ": " +
						types.elementAt (field) + NL;
				}
			}
		}
	catch (Exception exception)
		{
		description +=
			Description () + NL +
			"Contents could not be completed." + NL +
			Database_Exception.exception_String (exception);
		}
	}
else
	description = "Not connected to a database.";
return description;
}

/**	Gets the list of accessible catalogs in the database.
<p>
	@return	A Vector of Strings naming all the catalogs in the database.
	@throws	Database_Exception	If the Data_Port is not open, or
		the database rejects the operation.
*/
public Vector Catalogs ()
	throws Database_Exception
{
//	Data_Port Implementation
if ((DEBUG & DEBUG_DESCRIPTION) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Catalogs");
if (is_Open ())
	{
	try
		{
		//	Get the catalog list from the database metadata.
		String
			results_column_name,
			column_name;
		ResultSet
			catalog_set;
		if (Treat_Schema_As_Catalog)
			{
			results_column_name = "TABLE_SCHEM";
			catalog_set = JDBC_Connection.getMetaData ().getSchemas ();
			}
		else
			{
 			results_column_name = "TABLE_CAT";
			catalog_set = JDBC_Connection.getMetaData ().getCatalogs ();
			}
		if ((DEBUG & DEBUG_DESCRIPTION) != 0)
			{
			System.out.println
				("    " + (Treat_Schema_As_Catalog ? "Schemas" : "Catalogs")
					+ " metadata results:");
			report_ResultSetMetaData (catalog_set);
			}
		column_name =
			column_name_for_ResultSet (results_column_name, catalog_set);
		if ((DEBUG & DEBUG_DESCRIPTION) != 0)
			System.out.println
				("    Getting names from ResultSet column \""
					+ column_name + '"');
		if (column_name == null)
			throw new Database_Exception
				("The database metadata does not contain the expected \""
					+ results_column_name + "\" column name.");
		Vector
			catalogs = new Vector ();
		while (catalog_set.next ())
			catalogs.add (catalog_set.getString (column_name));
		if ((DEBUG & DEBUG_DESCRIPTION) != 0)
			System.out.println
				("    " + catalogs.size () + " catalogs: " + catalogs + NL
				+"<<< JDBC_Data_Port.Catalogs");
		return catalogs;
		}
	catch (Exception exception)
		{
		throw new Database_Exception
			(
			Description () + NL
			+ "Unable to obtain the Catalogs list.",
			exception
			);
		}
	}
else
	throw new Database_Exception
		(
		Description () + NL
		+ "Unable to obtain the Catalogs list." + NL
		+ "Not connected to a database."
		);
}

/**	Obtains the list of accessible tables contained in a catalog.
<p>
	If the catalog name is null, the <code>{@link Database#CATALOG
	CATALOG}</code> from the Data_Port <code>Configuration</code> will
	be used. The catalog name may be in <i>catalog</i>.<i>table</i>
	format, in which case only the catalog portion of the name will be
	used.
<p>
	@param	catalog	The name of the database calolog to examine. If this
		is in {@link #Table_Reference(String, String) table reference}
		format only the catalog part will be used. If null or empty the
		<code>{@link Database#CATALOG CATALOG}</code> value from the
		{@link #Configuration() configuration}, if available, will be
		used.
	@return	A Vector of Strings naming all the tables in the catalog.
		If the catalog does not exist in the database, null will be
		returned. An empty Vector will be returned for a catalog
		that does not contain any tables.
	@throws	Database_Exception	If the Data_Port is not open, no
		catalog name is available, or the database rejects the
		operation.
*/
public Vector Tables
	(
	String	catalog
	)
	throws Database_Exception
{
//	Data_Port Implementation
if ((DEBUG & DEBUG_DESCRIPTION) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Tables: catalog " + catalog);
if (is_Open ())
	{
	//	Use the catalog name known to the database server.
	try {catalog = database_catalog_name ("Tables", catalog, false);}
	catch (Database_Exception exception)
		{
		throw new Database_Exception
			("Unable to obtain Tables for catalog \"" + catalog + "\".",
			exception
			);
		}
	if ((DEBUG & DEBUG_DESCRIPTION) != 0)
		System.out.println
			("    Getting tables for catalog " + catalog);

	try
		{
		//	Get the tables list from the database metadata.
		String
			database_catalog = catalog,
			database_schema = "%";
		if (Treat_Schema_As_Catalog)
			{
			/*
				Since schema are being treated as Database catalogs
				it is presumed that a connection has been made to a
				particular database catalog that can be identified.
			*/
			database_catalog = JDBC_Connection.getCatalog ();
			database_schema = catalog;
			}
		ResultSet
			table_set = JDBC_Connection.getMetaData ().getTables
				(database_catalog, database_schema, "%", null);
		String
			column_name = column_name_for_ResultSet ("TABLE_NAME", table_set);
		if ((DEBUG & DEBUG_DESCRIPTION) != 0)
			{
			report_ResultSetMetaData (table_set);
			System.out.println
				("    Getting names from ResultSet column \""
					+ column_name + '"');
			}
		if (column_name == null)
			throw new Database_Exception
				("The database metadata does not contain the expected "
					+ "\"TABLE_NAME\" column name.");
		Vector
			tables = new Vector ();
		while (table_set.next ())
			tables.add (table_set.getString (column_name));
		if ((DEBUG & DEBUG_DESCRIPTION) != 0)
			System.out.println
				("<<< JDBC_Data_Port.Tables:" + NL
				+"    " + tables.size () + " tables: " + tables);
		return tables;
		}
	catch (Exception exception)
		{
		throw new Database_Exception
			(
			Description () + NL
			+"Unable to obtain Tables for catalog \"" + catalog + "\".",
			exception
			);
		}
	}
else
	throw new Database_Exception
		(
		Description () + NL
		+"Unable to obtain Tables for catalog \"" + catalog + "\"." + NL
		+"Not connected to a database."
		);
}

/**	Gets the list of field names in a table on the database server.
<p>
	@param	table	The name of the table to be examined. If null, then
		the <code>{@link Database#TABLE TABLE}</code> from the
		<code>Configuration</code> will be used. If it is not in
		{@link #Table_Reference(String, String) table reference} format,
		the <code>{@link Database#CATALOG CATALOG}</code> will be used.
	@return	A Vector of field name Strings. If the table does not
		exist in the Database, a null will be returned.
	@throws	Database_Exception	If the Data_Port is not open, no
		catalog name is available, or the operation on the database
		server failed.
*/
public Vector Field_Names
	(
	String	table
	)
	throws Database_Exception
{
//	Data_Port Implementation
return Fields (table, "COLUMN_NAME");
}

/**	Gets the list of field data types in a table on the database
	server.
<p>
	@param	table	The name of the table to be examined. If null, then
		the <code>{@link Database#TABLE TABLE}</code> from the
		<code>Configuration</code> will be used. If it is not in
		{@link #Table_Reference(String, String) table reference} format,
		the <code>{@link Database#CATALOG CATALOG}</code> will be used.
	@return	A Vector of field name Strings. If the table does not
		exist in the Database, a null will be returned.
	@throws	Database_Exception	If the Data_Port is not open, no table
		name is available, or the operation on the database server
		failed.
*/
public Vector Field_Types
	(
	String	table
	)
	throws Database_Exception
{
//	Data_Port Implementation
return Fields (table, "TYPE_NAME");
}

/**	Obtains a list of field information for the table of a catalog.
<p>
	The specified field_info must be a tag corresponding to one of the
	field information names obtained from the JDBC getColumns method:
<p>
	<dl>
	<dt>TABLE_CAT (String)
		<dd>Table catalog (may be null)
	<dt>TABLE_SCHEM (String)
		<dd>Table schema (may be null)
	<dt>TABLE_NAME (String)
		<dd>Table name
	<dt>COLUMN_NAME (String)
		<dd>Column (i.e. field) name
	<dt>DATA_TYPE (short)
		<dd>SQL type from java.sql.Types
	<dt>TYPE_NAME (String)
		<dd>Data source dependent type name, for a UDT the type
			name is fully qualified.
	<dt>COLUMN_SIZE (int)
		<dd>Column size. For char or date types this is the maximum
			number of characters, for numeric or decimal types this
			is precision.
	<dt>BUFFER_LENGTH (unused)
	<dt>DECIMAL_DIGITS (int)
		<dd>The number of fractional digits.
	<dt>NUM_PREC_RADIX (int)
		<dd>Radix (typically either 10 or 2)
	<dt>NULLABLE (int)
		<dd>Is NULL allowed?
		<dl compact>
    		<dt>columnNoNulls
				<dd>- might not allow NULL values
    		<dt>columnNullable
				<dd>- definitely allows NULL values
    		<dt>columnNullableUnknown
				<dd>- nullability unknown
		</dl>
	<dt>REMARKS (String)
		<dd>Comment describing column (may be null)
	<dt>COLUMN_DEF (String)
		<dd>Default value (may be null)
	<dt>SQL_DATA_TYPE (unused)
	<dt>SQL_DATETIME_SUB (unused)
	<dt>CHAR_OCTET_LENGTH (int)
		<dd>For char types the maximum number of bytes in the
			column.
	<dt>ORDINAL_POSITION (int)
		<dd>Index of the column in the table (starting at 1).
	<dt>IS_NULLABLE (String)
		<dl compact>
			<dt>NO
				<dd>- column definitely does not allow NULL values
			<dt>YES
				<dd>- column might allow NULL values
			<dt>An empty string
				<dd>- nobody knows
		</dl>
	</dl>
<p>
	@param	table	The name of the table to be examined. If this is
		null then the <code>{@link Database#TABLE TABLE}</code>
		from the <code>Configuration</code> will be used. If it is not in
		{@link #Table_Reference(String, String) table reference} format,
		the <code>{@link Database#CATALOG CATALOG}</code> will be used.
	@param	field_info	The tag String for selecting field information.
	@return	A Vector of field information Strings, one per field. For
		info that is originally numeric (as indicated in the table) a
		String representation is returned.
	@throws	Database_Exception	If no catalog or table name is
		available, or the database server rejected the operation.
	@see	java.sql.DatabaseMetaData#getColumns(String, String, String, String)
*/
public Vector Fields
	(
	String	table,
	String	field_info
	)
	throws Database_Exception
{
//	Data_Port Implementation
if ((DEBUG & DEBUG_DESCRIPTION) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Fields: table " + table + NL
		+"    field info: " + field_info);
if (is_Open ())
	{
	//	Both catalog and table names are required.
	String
		catalog = null,
		catalog_table = null;

	//	Use the catalog and table names known to the database server.
	try
		{
		catalog = database_catalog_name ("Fields", Catalog_Name (table), false);
		catalog_table = database_table_name ("Fields", table, false);
		}
	catch (Database_Exception exception)
		{
		throw new Database_Exception
			("Unable to obtain the Fields information for table \"" +
				table + "\".",
			exception);
		}
	try
		{
		//	Get the field names from the database metadata.
		String
			database_catalog = catalog,
			database_schema = null;
		if (Treat_Schema_As_Catalog)
			{
			/*
				Since schema are being treated as Database catalogs
				it is presumed that a connection has been made to a
				particular database catalog that can be identified.
			*/
			database_catalog = JDBC_Connection.getCatalog ();
			database_schema = catalog;
			}
		if ((DEBUG & DEBUG_DESCRIPTION) != 0)
			System.out.println
				("    Getting columns for table "
					+ catalog + Component_Delimiter + catalog_table);
		ResultSet
			fields_set = JDBC_Connection.getMetaData ().getColumns
				(database_catalog, database_schema, catalog_table, "%");
		String
			column_name =
				column_name_for_ResultSet (field_info, fields_set);
		if ((DEBUG & DEBUG_DESCRIPTION) != 0)
			{
			report_ResultSetMetaData (fields_set);
			System.out.println
				("    Getting names from ResultSet column " + column_name);
			}
		if (column_name == null)
			throw new Database_Exception
				("The database metadata does not contain the expected \""
					+ field_info + "\" column name.");
		Vector
			fields = new Vector ();
		while (fields_set.next ())
			fields.add (fields_set.getString (column_name));
		if ((DEBUG & DEBUG_DESCRIPTION) != 0)
			System.out.println
				("<<< JDBC_Data_Port.Fields:" + NL
				+"    " + fields.size () + " fields: " + fields);
		return fields;
		}
	catch (Exception exception)
		{
		throw new Database_Exception
			(
			Description () + NL
			+"Unable to obtain the Fields information for table \"" +
				catalog + Component_Delimiter + catalog+table + "\".",
			exception
			);
		}
	}
else
	throw new Database_Exception
		(
		Description () + NL
		+"Unable to obtain the Fields information for table \""
			+ table + "\"." + NL
		+"Not connected to a database."
		);
}

/**	Gets a list of primary keys for a table.
<p>
	@param	table	The name of the table to be examined. If this is
		null then the <code>{@link Database#TABLE TABLE}</code>
		from the <code>Configuration</code> will be used. If it is not in
		{@link #Table_Reference(String, String) table reference} format,
		the <code>{@link Database#CATALOG CATALOG}</code> will be used.
	@return	A Vector of field name Strings.
	@throws	Database_Exception	If no catalog and table name is
		available, or the database server rejected the operation.
*/
public Vector Keys
	(
	String	table
	)
	throws Database_Exception
{
//	Data_Port Implementation
if ((DEBUG & DEBUG_DESCRIPTION) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Keys: table " + table);
if (is_Open ())
	{
	//	Both catalog and table names are required.
	String
		catalog = null,
		catalog_table = null;

	//	Use the catalog and table names known to the database server.
	try
		{
		catalog = database_catalog_name ("Keys", Catalog_Name (table), false);
		catalog_table = database_table_name ("Keys", table, false);
		}
	catch (Database_Exception exception)
		{
		throw new Database_Exception
			("Unable to obtain the Keys for table \"" + table + "\".",
			exception);
		}
	try
		{
		//	Get the primary keys from the database metadata.
		String
			database_catalog = catalog,
			database_schema = null;
		if (Treat_Schema_As_Catalog)
			{
			/*
				Since schema are being treated as Database catalogs
				it is presumed that a connection has been made to a
				particular database catalog that can be identified.
			*/
			database_catalog = JDBC_Connection.getCatalog ();
			database_schema = catalog;
			}
		if ((DEBUG & DEBUG_DESCRIPTION) != 0)
			System.out.println
				("    Getting primary keys for table "
					+ catalog + Component_Delimiter + catalog_table);
		ResultSet
			keys_set = JDBC_Connection.getMetaData ().getPrimaryKeys
				(database_catalog, database_schema, catalog_table);
		String
			column_name =
				column_name_for_ResultSet ("COLUMN_NAME", keys_set);
		if ((DEBUG & DEBUG_DESCRIPTION) != 0)
			System.out.println
				("    Getting names from ResultSet column \""
					+ column_name + '"');
		if (column_name == null)
			throw new Database_Exception
				("The database metadata does not contain the expected \""
					+ "\"COLUMN_NAME\" column name.");
		Vector
			keys = new Vector ();
		while (keys_set.next ())
			keys.add (keys_set.getString (column_name));
		if ((DEBUG & DEBUG_DESCRIPTION) != 0)
			System.out.println
				("<<< JDBC_Data_Port.Keys:" + NL
				+"    " + keys.size () + " keys: " + keys);
		return keys;
		}
	catch (Exception exception)
		{
		throw new Database_Exception
			(
			Description () + NL
			+"Unable to obtain the Keys for table \"" +
				catalog + Component_Delimiter + catalog_table + "\".",
			exception
			);
		}
	}
else
	throw new Database_Exception
		(
		Description () + NL
		+"Unable to obtain the Keys for table \"" + table + "\"." + NL
		+"Not connected to a database."
		);
}

/*------------------------------------------------------------------------------
	Query
*/
/**	Submit an SQL query.
<p>
	The SQL_query String is an SQL statment that requests data to be
	extracted (not removed) from the database. The syntax, though
	"standardized", is likely to vary in its particulars dependent
	on the database server implementation.
<p>
	A query always returns a table. The table is in the form of a
	Vector of data record Vectors. The first record contains the names
	(String) of each field for the data in corresponding locations of
	all the field value records that follow; i.e. these are the table
	column names. The field values in each data record are always
	provided in String representation regardless of the data type
	stored in the database. However, NULL field values remain unchanged
	as they are distinct from an otherwise valid field value.
<p>
	<b>Note</b>: A field value will either be a String or null.
<p>
	<b>N.B.</b>: All {@link #Add_SQL_Listener(SQL_Listener) SQL statement
	listeners} are notified before the SQL is sent to the database server.
<p>
	@param	SQL_query	The syntax of the query string is database dependent.
		A typical example is an SQL "SELECT" statement.
	@param	limit	The maximum number of records to return. If
		negative, there will be no limit to the number of records
		returned. If zero, no records will be returned.
	@return	A data table in form of a Vector of String Vectors (records).
		The first record is the field names for the field values in all
		subsequent records.
	@throws	Database_Exception	If the Data_Port is not open or the
		operation on the database server failed.
*/
public Vector Query
	(
	String	SQL_query,
	int		limit
	)
	throws Database_Exception
{
//	Data_Port Implementation
if ((DEBUG & DEBUG_QUERY) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Query:" + NL
		+"    SQL - " + SQL_query);
Report_SQL_Statement (SQL_query);
if (is_Open ())
	{
	try
		{
		//	Submit the SQL query.
		Statement
			statement = JDBC_Connection.createStatement ();
		ResultSet
			results = statement.executeQuery (SQL_query);

		//	Assemble the results into a Vector of String Vector records.
		ResultSetMetaData
			results_metadata = results.getMetaData ();
		int
			total_fields = results_metadata.getColumnCount ();
		Vector
			table = new Vector (),
			field_names = new Vector (total_fields),
			record;

		//	Field names record.
		for (int
				field = 1;
				field <= total_fields;
				field++)
			field_names.add (results_metadata.getColumnName (field));
		table.add (field_names);

		//	Data records.
		while (limit-- != 0 &&
				results.next ())
			{
			record = new Vector (total_fields);
			for (int
					field = 1;
					field <= total_fields;
					field++)
				record.add (results.getString (field));
			table.add (record);
			}

		statement.close ();
		if ((DEBUG & DEBUG_QUERY) != 0)
			System.out.println
				("<<< JDBC_Data_Port.Query");
		return table;
		}
	catch (Exception exception)
		{
		throw new Database_Exception
			(
			Description () + NL
			+"Query failed for -" + NL
			+ SQL_query,
			exception
			);
		}
	}
else
	throw new Database_Exception
		(
		Description () + NL
		+"Query failed for -" + NL
		+ SQL_query + NL
		+"Not connected to a database."
		);
}

/**	Selects from one or more tables the data values from one or more
	named fields from the data records that satisfy the selection
	conditions.
<p>
	<b>Note</b>: Each table name that is not in the format
	<i>catalog</i>.<i>table</i> will have the <code>{@link
	Database#CATALOG CATALOG}</code> from the <code>Configuration</code>
	prepended.
<p>
	<b>N.B.</b>: If the limit argument is zero the SQL assmebled for the
	database query will be returned instead of submitting the query.
<p>
	@param	tables	The Vector of database tables from which to select
		records. If this is null, the tables (one or more) specified by
		the <code>{@link Database#TABLE TABLE}</code> parameter
		will be used.
	@param	fields	A Vector of field name Strings specifying the data
		fields (columns) to be extracted. If this is null or empty, all
		data fields will be extracted.
	@param	conditions	A String in SQL WHERE clause syntax
		(without the keyword "WHERE") specifying the conditions for
		selection of a data record from the database. The specific
		syntax of the conditions string is database dependent. If
		this is null, no conditions will be applied; all data records
		will be used.
	@param	limit	The maximum number of records to return. If negative,
		there will be no limit to the number of records returned.
		<b>N.B.</b>: If zero, the returned vector will contain a single
		string that is the SQL statement that would have been submitted
		to the Query method.
	@return	A Vector of String Vectors data table,as provided by the
		<code>{@link #Query(String, int) Query}</code> method.
	@throws	Database_Exception	If the Data_Port is not open, no table
		name is available, or the operation on the database server
		failed.
*/
public Vector Select
	(
	Vector	tables,
	Vector	fields,
	String	conditions,
	int		limit
	)
	throws Database_Exception
{
//	Data_Port Implementation
if ((DEBUG & DEBUG_QUERY) != 0)
	System.out.println
		(">-< JDBC_Data_Port.Select:" + NL
		+"    tables: " + tables + NL
		+"    fields: " + fields + NL
		+"    conditions: " + conditions);
String
	query = "SELECT ",
	list = List_String (fields);
if (list.length () == 0)
	list = "*";
query += list;

String
	catalog = Config_Value (Database.CATALOG);

if (tables == null ||
	tables.isEmpty ())
	tables = The_Configuration.Get_Linked (Database.TABLE);
if (tables != null &&
	catalog.length () != 0)
	{
	//	Apply the default catalog.
	ListIterator
		table_list = tables.listIterator ();
	while (table_list.hasNext ())
		{
		String
			table = (String)table_list.next ();
		if (table.length () != 0 &&
			table.indexOf (Component_Delimiter) < 0)
			table_list.set (catalog + Component_Delimiter + table);
		}
	}		
list = List_String (tables);
if (list.length () == 0)
	throw new Database_Exception
		(
		Description () + NL
		+"Unable to Select without a table specified." + NL
		+"This can be provided with a \"" + Database.TABLE +
			"\" Configuration parameter."
		);

query += " FROM " + list;

if (conditions != null &&
	conditions.length () != 0)
	query += " WHERE " + conditions;

if (limit == 0)
	{
	Vector
		SQL_statement = new Vector (1);
	SQL_statement.add (query);
	return SQL_statement;
	}
return Query (query, limit);
}

/*------------------------------------------------------------------------------
	Update
*/
/**	Submit an SQL update.
<p>
	The syntax of the SQL_update String is database dependent. These
	operations modify the database and return a count of the number of
	modifications, which may be 0 in some cases (e.g. when a catalog or
	table is created). Typical examples are SQL "UPDATE", "INSERT", and
	"ALTER" statements.
<p>
	<b>N.B.</b>: All {@link #Add_SQL_Listener(SQL_Listener) SQL statement
	listeners} are notified before the SQL is sent to the database server.
<p>
	@param	SQL_update	The SQL_update statement.
	@return	A count of the number of modifications.
	@throws	Database_Exception	If the Data_Port is not open or the
		operation on the database server failed.
*/
public int Update
	(
	String	SQL_update
	)
	throws Database_Exception
{
//	Data_Port Implementation
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Update: SQL -" + NL
		+ SQL_update);
Report_SQL_Statement (SQL_update);
if (is_Open ())
	{
	try
		{
		Statement
			statement = JDBC_Connection.createStatement ();
		int
			count = statement.executeUpdate (SQL_update);
		statement.close ();
		if ((DEBUG & DEBUG_UPDATE) != 0)
			System.out.println
				("<<< JDBC_Data_Port.Update: " + count);
		return count;
		}
	catch (Exception exception)
		{
		throw new Database_Exception
			(
			Description () + NL
			+"Update failed for -" + NL
			+ SQL_update,
			exception
			);
		}
	}
else
	throw new Database_Exception
		(
		Description () + NL
		+"Update failed for -" + NL
		+ SQL_update + NL
		+"Not connected to a database."
		);
}

/*..............................................................................
	Catalogs:
*/
/**	Creates a catalog in the database.
<p>
	@param	catalog	The name of the catalog to create. If it is in
		{@link #Table_Reference(String, String) table reference} format,
		the catalog part will be used.
	 A null catalog
		name, or a name of a catalog already in the database, is
		ignored.
	@throws	Database_Exception	If the database rejected the operation.
*/
public void Create
	(
	String	catalog
	)
	throws Database_Exception
{
//	Data_Port Implementation
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Create: catalog " + catalog);
String
	name = Catalog_Name (catalog);
if (name.length () > 0)
	catalog = name;
try
	{
	if (catalog != null &&
		! Database.Matches
			(Catalogs (), catalog, Case_Sensitive_Identifiers))
		Update ("CREATE "
			+ (Treat_Schema_As_Catalog ?
				Database_Schema_Term : Database_Catalog_Term)
			+ " " + catalog);
	}
catch (Database_Exception exception)
	{
	throw new Database_Exception
		(
		"Unable to Create catalog \"" + catalog + "\".",
		exception
		);
	}
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		("<<< JDBC_Data_Port.Create");
}

/**	Deletes a catalog from the database.
<p>
	@param	catalog	The name of the catalog to delete. A null catalog
		name, or a name that is not a catalog in the database, is
		ignored.
	@throws	Database_Exception	If the database rejected the operation.
*/
public void Delete
	(
	String	catalog
	)
	throws Database_Exception
{
//	Data_Port Implementation
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Delete: catalog " + catalog);
String
	name = Catalog_Name (catalog);
if (name.length () > 0)
	catalog = name;
try
	{
	if (catalog != null && Database.Matches
			(Catalogs (), catalog, Case_Sensitive_Identifiers))
		Update ("DROP "
			+ (Treat_Schema_As_Catalog ?
				Database_Schema_Term : Database_Catalog_Term)
			+ " " + catalog);
	}
catch (Database_Exception exception)
	{
	throw new Database_Exception
		(
		"Unable to Delete catalog \"" + catalog + "\".",
		exception
		);
	}
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		("<<< JDBC_Data_Port.Delete");
}

/*..............................................................................
	Tables:
*/
/**	Creates data tables with their fields and the data types of the fields.
<p>
	If the table does not exist, a new one is created. For each name
	String in the fields Vector, if the field does not exist, it is
	created with the data type from the corresponding entry in the
	types Vector. If the field is already present in the table, its
	data type is changed if needed.
<p>
	A table name that is not in the format <i>catalog</i>.<i>table</i>
	will have the <code>{@link Database#CATALOG CATALOG}</code>
	from the <code>Configuration</code> prepended. If the catalog does
	not exist in the database, it is created.
<p>
	@param	table	The name of the table to be affected. If null, the
		<code>{@link Database#TABLE TABLE}</code> from the
		<code>Configuration</code>, if any, will be used. If it is not in
		{@link #Table_Reference(String, String) table reference} format,
		the <code>{@link Database#CATALOG CATALOG}</code> will be used.
	@param	fields	The Vector of field names to be used. If null,
		nothing is done.
	@param	types	The Vector of data type names to be applied to
		the corresponding fields. If null, nothing is done.
	@throws	Database_Exception	If the Data_Port is not open, the
		number of fields and types are not the same, no table name is
		available, or the operation on the database server failed.
*/
public void Create
	(
	String	table,
	Vector	fields,
	Vector	types
	)
	throws Database_Exception
{
//	Data_Port Implementation
if (fields == null ||
	fields.isEmpty () ||
	types == null  ||
	types.isEmpty ())
	//	Ignored: nothing to do.
	return;
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Create: table " + table + NL
		+"    fields: " + fields + NL
		+"     types: " + types);
if (fields.size ()
	!= types.size ())
	throw new Database_Exception
		(
		Description () + NL
		+"Unable to Create table \"" + table + "\"." + NL
		+"There " + ((fields.size () == 1) ? "is" : "are") + fields.size ()
			+ " field" + ((fields.size () == 1) ? " " : "s ")
			+ "but " + types.size ()
			+ " type" + ((types.size () == 1) ? "." : "s.")
		);

//	Both a catalog and table name are required.
String
	catalog = catalog_name ("Create", table);
table = table_name ("Create", table);
Vector
	tables;
try {tables = Tables (catalog);}
catch (Database_Exception exception)
	{
	throw new Database_Exception
		(
		"Unable to Create table \"" + table + "\".",
		exception
		);
	}
if (Database.Matches (tables, table, Case_Sensitive_Identifiers))
	{
	//	Alter an existing table.
	table = catalog + Component_Delimiter + table;
	String
		alter = "ALTER TABLE " + table;
	try
		{
		Vector
			field_names = Field_Names (table),
			field_types = Field_Types (table);
		Iterator
			field_list = fields.iterator (),
			type_list = types.iterator ();
		while (field_list.hasNext ())
			{
			String
				field = (String)field_list.next (),
				type = (String)type_list.next ();
			int
				index = Database.Index
							(field_names, field, Case_Sensitive_Identifiers);
			if (index >= 0)
				{
				//	Existing field.
				if (! type.equalsIgnoreCase ((String)field_types.get (index)))
					//	Change the field type to that specified.
					Update (alter + " MODIFY " + field + " " + type);
				}
			else
				//	Add a new field.
				Update (alter + " ADD " + field + " " + type);
			}
		}
	catch (Database_Exception exception)
		{
		throw new Database_Exception
			(
			"Unable to Create (alter) table \"" + table + "\".",
			exception
			);
		}
	}
else
	{
	//	Create a new table.
	if ((DEBUG & DEBUG_UPDATE) != 0)
		System.out.println
			("    CREATE TABLE");
	try
		{
		//	Ensure that the catalog exists.
		Create (catalog);
		String
			create = "CREATE TABLE " +
				catalog + Component_Delimiter + table + " (";
		Iterator
			field_list = fields.iterator (),
			type_list  = types.iterator ();
		String
			delimiter = null;
		while (field_list.hasNext ())
			{
			if (delimiter == null)
				delimiter = ", ";
			else
				create += delimiter;
			create +=
				(String)field_list.next () + " " +
				(String)type_list.next ();
			}
		Update (create + ")");
		}
	catch (Database_Exception exception)
		{
		throw new Database_Exception
			(
			"Unable to Create table \"" + table + "\".",
			exception
			);
		}
	}
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		("<<< JDBC_Data_Port.Create");
}

/**	Deletes fields from a data table, or the entire table.
<p>
	Each field in the fields list is deleted from the table. If
	the field is not present in the table, it is ignored. If the
	fields list is null, the entire table is deleted from the
	catalog. <b>Note</b>: A table is not deleted even if all of
	its fields are deleted. If the table does not exist in the
	database, nothing is done.
<p>
	@param	table	The name of the table to be affected. If this is null
		then the <code>{@link Database#TABLE TABLE}</code> from the
		<code>Configuration</code> will be used. If it is not in {@link
		#Table_Reference(String, String) table reference} format, the
		<code>{@link Database#CATALOG CATALOG}</code> will be used.
	@param	fields	The Vector of field names to be used.
	@throws	Database_Exception	If the Data_Port is not open, no table
		name is available, or the operation on the database server
		failed.
*/
public void Delete
	(
	String	table,
	Vector	fields
	)
	throws Database_Exception
{
//	Data_Port Implementation
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Delete: table " + table + NL
		+"    fields: " + fields);
//	Both a catalog and table name are required.
String
	catalog = catalog_name ("Delete", table);
table = table_name ("Delete", table);

Vector
	tables;
try {tables = Tables (catalog);}
catch (Database_Exception exception)
	{
	throw new Database_Exception
		(
		"Unable to Delete table \"" + table + "\".",
		exception
		);
	}
if (Database.Matches (tables, table, Case_Sensitive_Identifiers))
	{
	table = catalog + Component_Delimiter + table;
	if (fields == null)
		{
		try {Update ("DROP TABLE " + table);}
		catch (Database_Exception exception)
			{
			throw new Database_Exception
				(
				"Unable to Delete table \"" + table + "\".",
				exception
				);
			}
		}
	else
		{
		String
			delete = "ALTER TABLE " + table + " DROP ";
		try
			{
			Vector
				field_names = Field_Names (table);
			Iterator
				field_list = fields.iterator ();
			while (field_list.hasNext ())
				{
				String
					field = (String)field_list.next ();
				if (Database.Matches
						(field_names, field, Case_Sensitive_Identifiers))
					Update (delete + field);
				}
			}
		catch (Database_Exception exception)
			{
			throw new Database_Exception
				(
				"Unable to Delete a field from table \"" + table + "\".",
				exception
				);
			}
		}
	}
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		("<<< JDBC_Data_Port.Delete");
}

/**	Renames a table.
<p>
	@param	table	The name of the table to be affected. If this is null
		then the <code>{@link Database#TABLE TABLE}</code> from the
		<code>Configuration</code> will be used. If it is not in {@link
		#Table_Reference(String, String) table reference} format, the
		<code>{@link Database#CATALOG CATALOG}</code> will be used.
	@param	name	The new name for the table. If this is null,
		or matches the table name, nothing is done.
	@throws	Database_Exception	If no catalog or table name is
		available, or the database server rejected the operation.
*/
public void	Rename
	(
	String	table,
	String	name
	)
	throws Database_Exception
{
//	Data_Port Implementation
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Rename: " + table + " to " + name);
if (name != null &&
	! Database.Matches (table, name, Case_Sensitive_Identifiers))
	{
	//	Both catalog and table names are required.
	String
		catalog = catalog_name ("Rename table", table);
	table = table_name ("Rename table", table);
	try
		{
		if (Database.Matches
				(Tables (catalog), table, Case_Sensitive_Identifiers))
			Update ("ALTER TABLE "
				+ catalog + Component_Delimiter + table + " RENAME " 
				+ catalog + Component_Delimiter + name);
		}
	catch (Database_Exception exception)
		{
		throw new Database_Exception
			(
			"Unable to Rename table \"" + table + "\" to \"" + name + "\".",
			exception
			);
		}
	}
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println ("<<< JDBC_Data_Port.Rename");
}

/**	Renames fields in a table.
<p>
	Each field name in the fields Vector that exists in the table is
	renamed to the corresponding name in the names Vector. Field names
	that do not exist in the table are ignored.
<p>
	<b>N.B.</b>: The existing type for the field, obtained from the
	<code>{@link #Field_Types(String) Field_Types}</code> method, is
	re-applied to the field (due to requirements of the SQL syntax).
	This could alter other characteristics of the field.
<p>
	@param	table	The name of the table to be affected. If this is null
		then the <code>{@link Database#TABLE TABLE}</code> from the
		<code>Configuration</code> will be used. If it is not in {@link
		#Table_Reference(String, String) table reference} format, the
		<code>{@link Database#CATALOG CATALOG}</code> will be used.
	@param	fields	The Vector of field names to be renamed. If null,
		nothing is done.
	@param	names	The Vector of new field names. If null, nothing
		is done.
	@throws	Database_Exception	If the Data_Port is not open, the
		number of fields and types are not the same, no table name is
		available, or the operation on the database server failed.
*/
public void	Rename
	(
	String	table,
	Vector	fields,
	Vector	names
	)
	throws Database_Exception
{
//	Data_Port Implementation
if (fields == null || fields.isEmpty () ||
	names == null  || names.isEmpty ())
	//	Ignored: nothing to do.
	return;
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Rename: fields for table " + table + NL
		+"    from: " + fields + NL
		+"      to: " + names);
if (fields.size () != names.size ())
	throw new Database_Exception
		(
		Description () + NL
		+"Unable to Rename the fields of table \"" + table + "\"." + NL
		+"There " + ((fields.size () == 1) ? "is" : "are") + fields.size ()
			+ " field" + ((fields.size () == 1) ? " " : "s ")
			+ "but " + names.size () + " name"
			+ ((names.size () == 1) ? "." : "s.")
		);
//	Both catalog and table names are required.
table = composite_name ("Rename fields", table);
try
	{
	Vector
		field_names = Field_Names (table),
		field_types = Field_Types (table);
	if (field_types != null &&
		field_types.size () != 0)
		{
		Iterator
			field_list = fields.iterator (),
			name_list = names.iterator ();
		while (field_list.hasNext ())
			{
			String
				field = (String)field_list.next (),
				name = (String)name_list.next ();
			int
				index = Database.Index
							(field_names, field, Case_Sensitive_Identifiers);
			if (index >= 0)
				Update ("ALTER TABLE "
					+ table + " CHANGE " + field + " " + name + " "
					+ (String)field_types.get (index));
			}
		}
	}
catch (Database_Exception exception)
	{
	throw new Database_Exception
		(
		"Unable to Rename fields of table \"" + table + "\".",
		exception
		);
	}
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println ("<<< JDBC_Data_Port.Rename (fields)");
}

/*..............................................................................
	Fields:
*/
/**	Inserts values for selected fields into a table.
<p>
	A new data record is created in the table. The fields Vector lists
	the names of the fields to be assigned a data value from the
	corresponding entry in the values Vector. The Vector of data values
	does not necessarily have to contain Strings, as long as each
	object's toString method produces a valid representation of the field
	value (or the object is null). Though the fields and values Vectors
	must be the same size, not all fields in the table need to be
	included; the database is expected to provide a default value for
	missing fields.
<p>
	@param	table	The name of the table to be affected. If null, the
		<code>{@link Database#TABLE TABLE}</code> from the
		<code>Configuration</code> will be used. If it is not in
		{@link #Table_Reference(String, String) table reference} format,
		the <code>{@link Database#CATALOG CATALOG}</code> will be used.
	@param	fields	The Vector of field names for the data values. If null,
		nothing is done.
	@param	values	The Vector of data values corresponding the Vector
		of field names. If null, nothing is done.
	@return	The number of records inserted; 1 if successful; 0 otherwise.
	@throws	Database_Exception	If the Data_Port is not open, the number
		of fields and values are not the same, no table name is
		available, a specified field is not in the table, or the
		operation on the database server failed.
*/
public int Insert
	(
	String	table,
	Vector	fields,
	Vector	values
	)
	throws Database_Exception
{
//	Data_Port Implementation
if (fields == null ||
	fields.isEmpty () ||
	values == null ||
	values.isEmpty ())
	//	Ignored: nothing to do.
	return 0;
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Insert: into table " + table + NL
		+"    fields: " + fields + NL
		+"    values: " + values);
if (fields.size () != values.size ())
	throw new Database_Exception
		(
		Description () + NL
		+"Unable to Insert into table \"" + table + "\"." + NL
		+"There " + ((fields.size () == 1) ? "is" : "are") + fields.size ()
			+ " field" + ((fields.size () == 1) ? " " : "s ")
			+ "but " + values.size ()
			+ " value" + ((values.size () == 1) ? "." : "s.")
		);
//	Both catalog and table names are required.
table = composite_name ("Insert", table);
int
	count = 0;
try
	{
	Vector
		field_names = Field_Names (table),
		field_types = Field_Types (table);
		
	if (field_types != null &&
		field_types.size () != 0)
		{
		String
			insert = "INSERT INTO " + table +
				" (" + List_String (fields) + ") VALUES (",
			delimiter = null;
		Iterator
			field_list = fields.iterator (),
			value_list = values.iterator ();
		while (field_list.hasNext ())
			{
			String
				field = (String)field_list.next ();
			int
				index = Database.Index
							(field_names, field, Case_Sensitive_Identifiers);
			if (index < 0)
				throw new Database_Exception
					(
					Description () + NL
					+"No \"" + field + "\" field in the table."
					);
			if (delimiter == null)
				delimiter = ", ";
			else
				insert += delimiter;
			Object
				value = value_list.next ();
			insert += Database.Value_Syntax
				(((value == null) ? (String)null : value.toString ()),
				(String)field_types.get (index));
			}
		count = Update (insert + ')');
		}
	}
catch (Database_Exception exception)
	{
	throw new Database_Exception
		(
		"Unable to Insert into table \"" + table + "\".",
		exception
		);
	}
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		("<<< JDBC_Data_Port.Insert: " + count);
return count;
}

/**	Updates values for selected fields into a table.
<p>
	The fields Vector lists the names of the fields to be assigned a new
	data value from the corresponding entry in the values Vector. The
	Vector of data values does not necessarily have to contain Strings,
	as long as each object's toString method produces a valid
	representation of the field value (or the object is null). The
	records to have their field values updated will be selected by the
	conditions expression. This is an SQL WHERE expression (without the
	"WHERE" keyword). If no conditions are provided, then all records
	will be updated.
<p>
	@param	table	The name of the table to be affected. If this is
		null then the <code>{@link Database#TABLE TABLE}</code>
		from the <code>Configuration</code> will be used. If it is not in
		{@link #Table_Reference(String, String) table reference} format,
		the <code>{@link Database#CATALOG CATALOG}</code> will be used.
	@param	fields	The Vector of field names for the data values.
	@param	values	The Vector of data values corresponding the Vector
		of field names.
	@return	The number of records that were modified.
	@throws	Database_Exception	If the Data_Port is not open, the
		number of fields and values are not the same, no table name is
		available, or the operation on the database server failed.
*/
public int Update
	(
	String	table,
	Vector	fields,
	Vector	values,
	String	conditions
	)
	throws Database_Exception
{
//	Data_Port Implementation
if (fields == null || fields.isEmpty () ||
	values == null || values.isEmpty ())
	//	Ignored: nothing to do.
	return 0;
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Update: table " + table + NL
		+"    fields: " + fields + NL
		+"    values: " + values + NL
		+"    conditions: " + conditions);
if (fields.size () != values.size ())
	throw new Database_Exception
		(
		Description () + NL
		+"Unable to Update table \"" + table + "\"." + NL
		+"There " + ((fields.size () == 1) ? "is" : "are") + fields.size ()
			+ " field" + ((fields.size () == 1) ? " " : "s ")
			+ "but " + values.size () + " value"
			+ ((values.size () == 1) ? "." : "s.")
		);
//	Both catalog and table names are required.
table = composite_name ("Update", table);
int
	count = 0;
try
	{
	Vector
		field_names = Field_Names (table),
		field_types = Field_Types (table);
	if (field_types != null &&
		field_types.size () != 0)
		{
		String
			update = "UPDATE " + table + " SET ",
			delimiter = null;
		Iterator
			field_list = fields.iterator (),
			value_list = values.iterator ();
		while (field_list.hasNext ())
			{
			String
				field = (String)field_list.next ();
			int
				index = Database.Index
							(field_names, field, Case_Sensitive_Identifiers);
			if (index < 0)
				throw new Database_Exception
					(
					Description () + NL
					+"No \"" + field + "\" field to Update."
					);
			if (delimiter == null)
				delimiter = ", ";
			else
				update += delimiter;
			Object
				value = value_list.next ();
			update += field + "=" + Database.Value_Syntax
				(((value == null) ? (String)null : value.toString ()),
				(String)field_types.get (index));
			}

		if (conditions != null && conditions.length () != 0)
			update += " WHERE " + conditions;

		count = Update (update);
		}
	}
catch (Database_Exception exception)
	{
	throw new Database_Exception
		(
		"Unable to Update table \"" + table + "\".",
		exception
		);
	}
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		("<<< JDBC_Data_Port.Update: " + count);
return count;
}

/**	Deletes selected records from a table.
<p>
	The records to be deleted will be selected by the conditions
	expression. This is an SQL WHERE expression (without the "WHERE"
	keyword).
<p>
	<b>Warning</b>: If no conditions are provided, then all records will
	be deleted leaving an empty table.
<p>
	@param	table	The name of the table to be affected. If this is null
		then the <code>{@link Database#TABLE TABLE}</code> from the
		<code>Configuration</code> will be used. If it is not in {@link
		#Table_Reference(String, String) table reference} format, the
		<code>{@link Database#CATALOG CATALOG}</code> will be used.
	@param	conditions	A String in SQL WHERE clause syntax
		(without the keyword "WHERE") specifying the conditions for
		selection of a data record from the database. The specific
		syntax of the conditions string is database dependent. If
		this is null, no conditions will be applied and all data records
		will be deleted.
	@return	The number of records that were deleted.
	@throws	Database_Exception	If the Data_Port is not open, no table
		name is available, or the operation on the database server
		failed.
*/
public int Delete
	(
	String	table,
	String	conditions
	)
	throws Database_Exception
{
//	Data_Port Implementation
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		(">>> JDBC_Data_Port.Delete: from table " + table + NL
		+"    conditions: " + conditions);
//	Both catalog and table names are required.
String
	catalog = catalog_name ("Delete records", table);
table = table_name  ("Delete records", table);
int
	count = 0;
try
	{
	if (Database.Matches
			(Tables (catalog), table, Case_Sensitive_Identifiers))
		{
		String
			delete = "DELETE FROM " + catalog + Component_Delimiter + table;
		if (conditions != null && conditions.length () != 0)
			delete += " WHERE " + conditions;
		count = Update (delete);
		}
	}
catch (Database_Exception exception)
	{
	throw new Database_Exception
		(
		"Unable to Delete records from table \"" + table + "\".",
		exception
		);
	}
if ((DEBUG & DEBUG_UPDATE) != 0)
	System.out.println
		("<<< JDBC_Data_Port.Delete: " + count);
return count;
}

/*..............................................................................
	Utility:
*/
/**	Generates a table reference from a catalog and table name.
<p>
	A table reference in a Data_Port method is a composite name
	combining a catalog name and a table name in the format:
<p>
	catalog{@link #Table_Reference_Component_Delimiter() delimiter}table
<p>
	If a non-null, non-empty catalog name is specified the catalog
	portion of the name (which may itself be a table reference) will
	override any catalog name portion that might be in a table name that
	is already in table reference format. For null or empty table or
	catalog names the value of the <code>{@link Database#CATALOG
	CATALOG}</code> and/or <code>{@link Database#TABLE TABLE}</code>
	Configuration parameters will be used.
<p>
	@param	catalog	The catalog name portion for the table reference. If
		it is in table reference format, the catalog portion will be
		used. If null or empty, the <code>{@link Database#CATALOG
		CATALOG}</code> value from the {@link #Configuration()
		configuration}, if available, will be used.
	@param	table	The table name portion for the table reference. If
		it is in table reference format, the table portion will be
		used. If null, the <code>{@link Database#TABLE TABLE}</code>
		value from the {@link #Configuration() configuration}, if
		available, will be used.
	@return	A table reference suitable for use as a <code>table</code>
		argument of a JDBC_Data_Port method.
	@throws	Database_Exception	If a table reference can not be formed.
*/
public String Table_Reference
	(
	String	catalog,
	String	table
	)
	throws Database_Exception
{
String
	name = table;
if (table == null || table.length () == 0)
	table = Config_Value (Database.TABLE);
String
	table_name = Table_Name (table);
if (table_name.length () > 0)
	{
	String
		catalog_name;
	if ((catalog_name = Catalog_Name (catalog)).length () > 0 ||
		(catalog_name = Table_Name (catalog)).length () > 0 ||
		(catalog_name = Catalog_Name (table)).length () > 0 ||
		(catalog_name = Config_Value (Database.CATALOG)).length () > 0)
		return catalog_name + Component_Delimiter + table_name;
	}
throw new Database_Exception
	(
	Description () + NL
	+"Unable to form a Table_Reference using catalog \"" + catalog +
	"\" and table \"" + name + "\"."
	);
}

/**	Gets the catalog part of a {@link #Table_Reference(String, String)
	table reference}.
<p>
	@param	table_reference	A String that may be a composite name
		combining a catalog name and a table name. May be null.
	@return	The catalog name portion of the table reference. If the
		table_reference is null or does not contain a {@link
		#Component_Delimiter} character, the empty String is returned.
	@see #Table_Reference(String, String)
*/
public String Catalog_Name
	(
	String	table_reference
	)
{
int
	index;
if (table_reference != null &&
	(index = table_reference.indexOf (Component_Delimiter)) >= 0)
	return table_reference.substring (0, index);
return "";
}

/**	Gets the name known to the database server for a catalog name.
<p>
	Only accessible catalogs can be identified. If the database server
	determines that the connection does not have permission to access
	the specified catalog then it can not be identified.
<p>
	<b>N.B.</b>: Identifying the catalog from the list of accessible
	catalogs is done using the {@link #Case_Sensitive_Identifiers() case
	sensitivity} of the Data_Port implementation. Therefore, if case
	sensitive matching is used this method can only return the specified
	catalog name or null; in this case this method is only useful for
	determining if a catalog is accessible or not.
<p>
	@param	catalog	The name of the database calolog to examine. If this
		is a {@link #Table_Reference(String, String) table reference}
		only the catalog part will be used. If null or empty the
		<code>{@link Database#CATALOG CATALOG}</code> value from the
		{@link #Configuration() configuration}, if available, will be
		used.
	@return	The name of the catalog as it is known in the database. This
		will be null if a non-null, non-empty catalog name could not be
		identified in the database.
	@throws	Database_Exception	If the Data_Port is not open, a catalog
		name was not provided, or the database rejects the operation.
*/
public String Database_Catalog_Name
	(
	String	catalog
	)
	throws Database_Exception
{
//	Data_Port Implementation
return database_catalog_name ("Database_Catalog_Name", catalog, true);
}


private String database_catalog_name
	(
	String	method,
	String	catalog,
	boolean	always_check
	)
	throws Database_Exception
{
if ((DEBUG & DEBUG_DESCRIPTION) != 0)
	System.out.println
		(">>> JDBC_Data_Port.database_catalog_name: " + catalog);
/*
	Try to get a catalog name from the argument String.
	If that fails use the entire argument String,
	unless it is empty in which case try to get a default
	catalog name from the configuration.
*/
String
	name = Catalog_Name (catalog);
if (name.length () > 0)
	catalog = name;
else
if (catalog == null ||
	catalog.length () == 0)
	//	Try to get a default catalog from the configuration.
	catalog = catalog_name (method, catalog);

if (always_check ||
	! Case_Sensitive_Identifiers)
	{
	if (is_Open ())
		{
		//	Get the catalog list from the database metadata.
		Vector
			catalogs = null;
		try {catalogs = Catalogs ();}
		catch (Exception exception)
			{
			throw new Database_Exception
				(
				"Unable to obtain the database catalog name for \""
					+ catalog + "\"." + NL
				+ "Unable to obtain the Catalogs list.",
				exception
				);
			}
		if ((DEBUG & DEBUG_DESCRIPTION) != 0)
			System.out.println
				("    " + catalogs.size () + " catalogs: " + catalogs);
		int
			index =
				Database.Index
					(catalogs, catalog, Case_Sensitive_Identifiers);
		if (index < 0)
			catalog = null;
		else
			catalog = (String)catalogs.get (index);
		}
	else
		throw new Database_Exception
			(
			Description () + NL
			+"Unable to obtain the database catalog name for \""
				+ catalog + "\"." + NL
			+"Not connected to a database."
			);
	}
if ((DEBUG & DEBUG_DESCRIPTION) != 0)
	System.out.println
		("<<< JDBC_Data_Port.database_catalog_name: " + catalog);
return catalog;
}

/**	Gets the table part of a {@link #Table_Reference(String, String)
	table reference}.
<p>
	@param	table_reference	A String that may be a composite name
		combining a catalog name and a table name. May be null.
	@return	The table name portion of the table reference. If the
		table_reference does not contain a {@link #Component_Delimiter}
		character, the entire String is returned. If it is null the empty
		String is returned.
	@see #Table_Reference(String, String)
*/
public String Table_Name
	(
	String	table_reference
	)
{
if (table_reference == null)
	return "";
int
	index = table_reference.indexOf (Component_Delimiter) + 1;
if (index > 0)
	return table_reference.substring (index);
return table_reference;
}

/**	Gets the name known to the database server for a table name.
<p>
	Only accessible catalogs can be identified. If the database server
	determines that the connection does not have permission to access
	the specified catalog then it can not be identified.
<p>
	<b>N.B.</b>: Identifying the catalog from the list of accessible
	catalogs is done using the {@link #Case_Sensitive_Identifiers() case
	sensitivity} of the Data_Port implementation. Therefore, if case
	sensitive matching is used this method can only return the specified
	catalog name or null; in this case this method is only useful for
	determining if a catalog is accessible or not.
<p>
	@param	table	The name of the table to identify in the database. If
		null, then the <code>{@link Database#TABLE TABLE}</code> value
		from the <code>Configuration</code>, if present, will be used. If
		it is not in <i>catalog</i>.<i>table</i> format, the <code>{@link
		Database#CATALOG CATALOG}</code> value, if present, will be used.
	@return	The name of the catalog as it is known in the database. This
		will be null if a catalog name could not be identified in the
		database.
	@throws	Database_Exception	If the Data_Port is not open, catalog and
		table names were not provided, or the database rejects the
		operation.
*/
public String Database_Table_Name
	(
	String	table
	)
	throws Database_Exception
{
//	Data_Port Implementation
return database_table_name ("Database_Table_Name", table, true);
}


private String database_table_name
	(
	String	method,
	String	table,
	boolean	always_check
	)
	throws Database_Exception
{
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println
		(">>> JDBC_Data_Port.database_table_name: " + table);
String
	catalog_table = table_name (method, table);

if (always_check ||
	! Case_Sensitive_Identifiers)
	{
	if (is_Open ())
		{
		//	Both catalog and table names are required.
		String
			catalog = catalog_name (method, table);
		try
			{
			//	Get the table list from the database metadata.
			if ((DEBUG & DEBUG_UTILITY) != 0)
				System.out.println
					("    Getting tables for catalog " + catalog);
			Vector
				tables = Tables (catalog);
			int
				index =
					Database.Index
						(tables, catalog_table, Case_Sensitive_Identifiers);
			if (index < 0)
				catalog_table = null;
			else
				catalog_table = (String)tables.get (index);
			}
		catch (Exception exception)
			{
			throw new Database_Exception
				("Unable to obtain the database table name for table \""
					+ table + "\" in catalog \"" + catalog + "\"." + NL
				+ "Unable to obtain the Tables list.",
				exception
				);
			}
		}
	else
		throw new Database_Exception
			(
			Description () + NL
			+ "Unable to obtain the database table name for table \""
				+ table + "\"." + NL
			+ "Not connected to a database."
			);
	}
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println
		("<<< JDBC_Data_Port.database_table_name: " + catalog_table);
return catalog_table;
}

/**	Gets the table reference component delimiter.
<p>
	@return	The portion of a table reference that delimits the
		catalog component from the table component.
	@see #Table_Reference(String, String)
*/
public String Table_Reference_Component_Delimiter ()
{return Component_Delimiter;}

/**	@see	Data_Port#Case_Sensitive_Identifiers()
*/
public boolean Case_Sensitive_Identifiers ()
{return Case_Sensitive_Identifiers;}

/**	@see	Data_Port#Case_Sensitive_Identifiers(boolean)
*/
public void Case_Sensitive_Identifiers
	(
	boolean		case_sensitive
	)
{Case_Sensitive_Identifiers = case_sensitive;}

/*==============================================================================
	Utility methods
*/
/**	Gets a Value for a Configuration Parameter.
<p>
	Only one Value is obtained; this will be the first
	Value when the Parameter is associated with an Array.
<p>
	@param	parameter	The pathname to the Parameter.
	@return	The String representation of the Parameter Value.
		If the named Parameter can not be found an empty
		String will be returned.
*/
protected String Config_Value
	(
	String	parameter
	)
{
if (The_Configuration == null)
	parameter = null;
else
	parameter = The_Configuration.Get_Linked_One (parameter);
return (parameter == null) ? "" : parameter;
}

/**	Gets a catalog name.
<p>
	If the catalog_table String contains a Catalog_Name, that is
	returned. Otherwise the Database.CATALOG parameter from the
	Configuration, if present, is returned. If no catalog name can be
	found a Database_Exception is thrown.
<p>
	@param	method	The name of the method that needs a catalog name.
		This name will be used in the Database_Exception message.
	@param	catalog_table	A String to be provided to Catalog_Name in
		an attempt to extract a catalog name portion.
	@return	The name of a catalog.
	@throws	Database_Exception	If no catalog name can be found.
*/
protected String catalog_name
	(
	String	method,
	String	catalog_table
	)
	throws Database_Exception
{
String
	name = catalog_table;
if ((catalog_table = Catalog_Name (catalog_table)).length () == 0 &&
	(catalog_table = Config_Value (Database.CATALOG)).length () == 0)
	throw new Database_Exception
		(
		Description () + NL
		+ method + " requires a catalog name (given \"" + name + "\")." + NL
		+"This can be provided with a \"" + Database.CATALOG +
			"\" Configuration parameter."
		);
return catalog_table;
}

/**	Gets a table name.
<p>
	If the catalog_table String contains a Table_Name, that is returned.
	Otherwise the Database.TABLE parameter from the Configuration, if
	present, is returned. If no table name can be found a
	Database_Exception is thrown.
<p>
	@param	method	The name of the method that needs a table name.
		This name will be used in the Database_Exception message.
	@param	catalog_table	A String to be provided to Table_Name in
		an attempt to extract a table name portion.
	@return	The name of a table.
	@throws	Database_Exception	If no table name can be found.
*/
protected String table_name
	(
	String	method,
	String	catalog_table
	)
	throws Database_Exception
{
String
	name = catalog_table;
if ((catalog_table = Table_Name (catalog_table)).length () == 0 &&
	(catalog_table = Table_Name (Config_Value (Database.TABLE))).length () == 0)
	throw new Database_Exception
		(
		Description () + NL
		+ method + " requires a table name (given \"" + name + "\")." + NL
		+"This can be provided with a \"" + Database.TABLE +
			"\" Configuration parameter."
		);
return catalog_table;
}

/**	Gets a composite name.
<p>
	A composite name is the {@link #catalog_name(String, String)} plus
	the {@link #table_name(String, String)} separated by the {@link
	#Component_Delimiter}. This is effectively the same as a {@link
	#Table_Reference(String, String)} but derived from a single
	catalog_table reference name.
<p>
	@param	method	The name of the method that needs a composite name.
		This name will be used in the Database_Exception message.
	@param	catalog_table	A String to be provided to catalog_name
		and table_name in
		an attempt to obtain catalog and table names.
	@return	A table reference string with both catalog an table name parts.
	@throws	Database_Exception	If no catalog or table name can be found.
*/
protected String composite_name
	(
	String	method,
	String	catalog_table
	)
	throws Database_Exception
{
return
	catalog_name (method, catalog_table) +
	Component_Delimiter +
	table_name (method, catalog_table);
}

/**	Produce a String representation of a Vector list.
<p>
	Each list element, which must be a String of non-zero length (zero
	length String elements are ignored), are concatenated with the ", "
	delimiter between them. This produces a list representation suitable
	for use in SQL syntax.
<p>
	@param	list	A Vector of Strings.
	@return	A String containing all the non-zero length elements of the
		list, in the order they occur in the list, separated by the ", "
		delimiter.
*/
protected static String List_String
	(
	Vector	list
	)
{
String
	list_string = "";

if (list != null && ! list.isEmpty ())
	{
	String
		element,
		delimiter = null;
	Iterator
		elements = list.iterator ();
	while (elements.hasNext ())
		{
		element = (String)elements.next ();
		if (element.length () != 0)
			{
			if (delimiter == null)
				delimiter = ", ";
			else
				list_string += delimiter;
			list_string += element;
			}
		}
	}
return list_string;
}


private static String column_name_for_ResultSet
	(
	String		column_name,
	ResultSet	result_set
	)
	throws SQLException
{
if (column_name == null ||
	result_set == null)
	return column_name;

ResultSetMetaData
	metadata = result_set.getMetaData ();
String
	name;
int
	columns = metadata.getColumnCount (),
	count = 0;
//	N.B.: The first column is number 1.
while (count++ < columns)
	if (column_name.equalsIgnoreCase (name = metadata.getColumnName (count)))
		return name;
return null;
}


private static void report_ResultSetMetaData
	(
	ResultSet	results
	)
	throws SQLException
{
ResultSetMetaData
	metadata = results.getMetaData ();
int
	columns = metadata.getColumnCount (),
	count = 0;
System.out.println
	("    ResultSetMetaData:" + NL
	+"    " + columns + " columns -");
//	N.B.: The first column is number 1.
while (count++ < columns)
	System.out.println
		("      Column " + count + NL
		+"      isCaseSensitive = " + metadata.isCaseSensitive (count) + NL
		+"          ColumnLabel = " + metadata.getColumnLabel (count) + NL
		+"           ColumnName = " + metadata.getColumnName (count) + NL
		+"           SchemaName = " + metadata.getSchemaName (count) + NL
		+"            TableName = " + metadata.getTableName (count));
}


}	//	End of class

