/*	PVL_to_DB

PIRL CVS ID: PVL_to_DB.java,v 1.39 2012/04/16 06:14:23 castalia Exp

Copyright (C) 2003-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.PVL;

//	PIRL packages
import PIRL.Configuration.Configuration;
import PIRL.Configuration.Configuration_Exception;
import PIRL.Database.*;
import PIRL.Conductor.*;
import PIRL.Strings.Words;
import PIRL.Strings.String_Buffer;
import PIRL.Strings.String_Utilities;
import PIRL.Utilities.Host;
import PIRL.Utilities.Streams;

//	Other packages
import	java.util.Vector;
import	java.util.Iterator;
import	java.io.File;
import	java.io.InputStream;
import	java.io.StringWriter;
import	java.text.ParseException;
import	java.lang.IllegalArgumentException;
import	java.lang.NumberFormatException;


/**	<i>PVL_to_DB</i> maps a source of parameter values to database
	table field values.
<p>
	A <i>Map</i> is used to define the assignment of PVL Parameter
	Values, or constant values, to Database fields. The map is defined
	using PVL Parameters. The Map defines the assignment from the value
	of a resolved source parameter reference to a resolved database
	field reference where the value is deposited. PVL_to_DB binds a PVL
	source of Parameter Values and a Database target to the Map. In
	addition, a set of configuration parameters may be specified that
	will supplement every PVL source mapped to the Database; these may
	be thought of as default parameters.
<p>
	The name of each Assignment Parameter in the map specifies a database
	reference to a single database field. The database reference may
	specify either an update of an existing record or an insert of a new
	record. A database reference may be flagged to try an update and, if
	this fails, to fallback to an insert. Field reference Assignment
	Parameters may be contained in an Aggregate Parameter for a single
	update or insert operation on the Database of the collected
	assignments.
<p>
	The value of each Parameter in the map specifies a parameter source
	reference. This is resolved against the PVL source, unless it is a
	constant value. A constant value may be a quoted string or the
	result of a mathematical expression that evaluates to a constant.
<p>
	Both database references and parameter source references may
	contain nested references of either type, which themselves may
	contain nested references.
<p>
	The map is processed in the order in which the parameters occur.
	The map may be processed in a loop until either no source
	references are resolved or a specified number of passes have been
	accomplished.
<p>
@author		Bradford Castalia, Christian Schaller - UA/PIRL
@version	1.39
@see		PIRL.PVL.Parameter
@see		PIRL.PVL.Value
@see		PIRL.Database.Database
@see		PIRL.Conductor.Reference_Resolver
*/
public class PVL_to_DB
	extends Update_DB
{
/**	The Class identification.
*/
public static final String
	ID = "PIRL.PVL.PVL_to_DB (1.39 2012/04/16 06:14:23)";

//	Configuration:

/**	The Parameter Group for this class: "PVL_to_DB".
<p>
	This group will be extracted from the Configuration parameters, or
	an empty group of this name will be created if it is not present in
	the Configuration. Then a set of common parameters are added.
	It is appended to the PVL source parameters.
*/
public static final String
	PVL_to_DB_GROUP						= "PVL_to_DB";
private Configuration
	PVL_to_DB_Group						= null;

/**	The name given to the {@link #Source_Parameters() Source_Parameters}
	when the Aggregate is based on an Assignment Parameter: "Source".
*/
public static final String
	DEFAULT_SOURCE_PARAMETERS_NAME		= "Source";

/**	The name of the Parameter in the PVL_to_DB group that has the
	system hostname: "Hostname".
*/
public static final String
	HOSTNAME_PARAMETER					= "Hostname";

/**	The name of the Parameter in the PVL_to_DB group that has the
	current working directory pathname: "CWD".
*/
public static final String
	CWD_PARAMETER						= "CWD";

/**	The name of the Parameter in the PVL_to_DB group that has the
	name of the current {@link #Source_Parameters() Source_Parameters}:
	"Source_Name".
*/
public static final String
	SOURCE_NAME_PARAMETER				= "Source_Name";

/**	The name of the Parameter in the PVL_to_DB group that has the
	absolute pathname of the current {@link #PVL_Source() PVL_Source} name:
	{@link Conductor#SOURCE_PATHNAME_PARAMETER
	SOURCE_PATHNAME_PARAMETER}.
*/
public static final String
	SOURCE_PATHNAME_PARAMETER
		= Conductor.SOURCE_PATHNAME_PARAMETER;

/**	The name of the Parameter in the PVL_to_DB group that has the
	directory pathname of the current {@link #PVL_Source() PVL_Source} name:
	{@link Conductor#SOURCE_DIRECTORY_PARAMETER
	SOURCE_DIRECTORY_PARAMETER}.
*/
public static final String
	SOURCE_DIRECTORY_PARAMETER
		= Conductor.SOURCE_DIRECTORY_PARAMETER;

/**	The name of the Parameter in the PVL_to_DB group that has the
	filename of the current {@link #PVL_Source() PVL_Source} name:
	{@link Conductor#SOURCE_FILENAME_PARAMETER
	SOURCE_FILENAME_PARAMETER}.
*/
public static final String
	SOURCE_FILENAME_PARAMETER
		= Conductor.SOURCE_FILENAME_PARAMETER;

/**	The name of the Parameter in the PVL_to_DB group that has the portion
	of the {@link #SOURCE_FILENAME_PARAMETER SOURCE_FILENAME_PARAMETER}
	value without the extension (the portion preceding the last period
	character): {@link Conductor#SOURCE_FILENAME_ROOT_PARAMETER
	SOURCE_FILENAME_ROOT_PARAMETER}.
*/
public static final String
	SOURCE_FILENAME_ROOT_PARAMETER
		= Conductor.SOURCE_FILENAME_ROOT_PARAMETER;

/**	The name of the Parameter in the PVL_to_DB group that has the portion
	of the {@link #SOURCE_FILENAME_PARAMETER SOURCE_FILENAME_PARAMETER}
	value following the last period ('.') character: {@link
	Conductor#SOURCE_FILENAME_PARAMETER SOURCE_FILENAME_PARAMETER}.
*/
public static final String
	SOURCE_FILENAME_EXTENSION_PARAMETER
		= Conductor.SOURCE_FILENAME_EXTENSION_PARAMETER;

/**	The name of the Parameter in the PVL_to_DB group that has the 
	current mapping loop counter: "Loop_Counter".
<p>
	The value of this parameter is incremented for each
	mapping {@link #Loop(boolean) Loop} through the {@link
	#PVL_Source() Source PVL} parameters. It is initialized to 0.
*/
public static final String
	LOOP_COUNTER_PARAMETER				= "Loop_Counter";

/**	The {@link #Loop(boolean) Loop} limit: 256.
<p>
	The maximum number of times that a mapping loop is allowed to
	occur.
*/
public int
	Loop_Limit							= 256;
private boolean
	Loop								= false;

//	Map:

/**	The Group name of the map parameters: "PVL_to_DB_Map".
*/
public static final String
	MAP_NAME							= "PVL_to_DB_Map";
private Parameter
	The_Map								= null;

//	Reference_Resolver
private Reference_Resolver
	The_Resolver						= null;

/**	Default value if a reference can not be resolved.
*/
public static String
	RESOLVER_DEFAULT_VALUE				= null;

//	Source Parameters:
private Parameter
	Source_Parameters					= null;

/**	Whether to retry failed resolving of Group name nested references
	with both configuration and source parameters.
*/
public boolean
	Retry_Group_with_Source				= true;

/**	Alternative source parameter reference token character.
*/
public static final char
	ALTERNATIVE_MARKER					= '|';

/**	Non-math expression source parameter reference token character.
*/
public static final char
	NO_MATH_MARKER						= ':';

/**	Update-or-insert database reference marker character.
<p>
	A database reference prefixed with this marker will be used for
	an insert operation if the initial update operation fails. If the
	database reference does not include a record key specification the
	marker will simply be ignored. <b>N.B.</b>: The marker is not part
	of the database reference; it is removed before the database
	reference is used.
*/
public static final char
	UPDATE_OR_INSERT_MARKER				= '*';

/**	Whether an unresolved source parameter reference will throw an
	Invalid_Map_Syntax exception by default.
*/
public static boolean
	UNRESOLVED_SOURCE_REFERENCE_THROWS	= false;

/**	The Configuration parameter which, if {@link
	Configuration#Enabled(String, boolean) enabled}, will cause any
	unresolved source parameter reference to throw an Invalid_Map_Syntax
	exception.
<p>
	If this parameter is not found, the {@link
	#UNRESOLVED_SOURCE_REFERENCE_THROWS} value will be used by default.
*/
public static final String
	UNRESOLVED_SOURCE_REFERENCE_PARAMETER
		= "Unresolved_Source_Reference_Throws";

private boolean
	Unresolved_Source_Reference_Throws	= UNRESOLVED_SOURCE_REFERENCE_THROWS;

//	No database operations.
private boolean
	No_Op								= false;



/**	Exit status: Success.
*/
public static final int
	EXIT_SUCCESS						= 0;

/**	Exit status: PVL processing problem.
*/
public static final int
	EXIT_PVL_ERROR						= 1;

/**	Exit status: Invalid syntax was found in the map.
*/
public static final int
	EXIT_INVALILD_SYNTAX				= 2;

/**	Exit status: An unresolved reference was encountered.
*/
public static final int
	EXIT_UNRESOLVED_REFERENCE			= 3;


//	Debug control
private static final int
	DEBUG_OFF					= 0,
	DEBUG_CONSTRUCTOR			= 1 << 0,
	DEBUG_CONFIGURATION			= 1 << 1,
	DEBUG_MAP					= 1 << 2,
	DEBUG_SOURCE				= 1 << 3,
	DEBUG_PARAMETER_REFERENCE	= 1 << 4,
	DEBUG_FIELD_REFERENCE		= 1 << 5,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs from a Database and a map source.
<P>
	@param	database	The Database to be target.
	@param	map_source	The source of the database map. May be
		a URL refering to an appropriate database map.
	@throws	Configuration_Exception	If there is a problem with the
		configuration file.
	@throws	Database_Exception	If there is a problem connecting with
		the database.
	@throws PVL_Exception		If there is a problem parsing the PVL from
		the database map file.
*/
public PVL_to_DB
	(
	Database	database,
	String		map_source
	)
	throws
		Configuration_Exception,
		PVL_Exception
{this (database, Assemble_PVL (map_source));}

/**	Constructs from a Database and a Parameter map Aggregate.
<p>
	The {@link Update_DB#Update_One_Only(boolean) update mode} is
	set to only allow one record per update operation.
<p>
	@param	database	The Database to be updated.
	@param	map		The Parameter map aggregate.
	@throws	Configuration_Exception	If there is a problem with the
		Database configuration.
*/
public PVL_to_DB
	(
	Database	database,
	Parameter	map
	)
	throws
		Configuration_Exception
{
//	Construct the base Update_DB object.
super (database);
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println (">>> PVL_to_DB:");

//	Default update mode.
Update_One_Only (true);

//	Construct the Reference_Resolver for the database.
The_Resolver = new Reference_Resolver (database);

//	Assign the PVL map.
The_Map = map;

//	Update the configuration.
Configure ();

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("<<< PVL_to_DB");
}

/*==============================================================================
	Modes
*/
/**	Enables or disables database update operations.
<p>
	As an aid in debugging complex references in map parameter syntax
	final database update operations may be disabled so that the results
	of resolving map parameter references may be viewed without affecting
	the database contents. This mode is implicitly
	{@link Update_DB#Verbose(boolean) verbose}.
<p>
	<b>Note</b>: No-op mode is disabled by default.
<p>
	@param	enable	true if no-op mode is to be enabled (database
		update operations are disabled); false otherwise.
	@return	This PVL_to_DB object.
*/
public PVL_to_DB No_Op
	(
	boolean	enable
	)
{
No_Op = enable;
return this;
}

/**	Tests if no-op mode is enabled.
<p>
	@return	true if no-op mode is enabled; false otherwise.
	@see	#No_Op(boolean)
*/
public boolean No_Op ()
{return No_Op;}

/**	Enables or disables map looping.
<p>
	When enabled a (@link #PVL_Source() PVL_Source} is repeatedly
	mapped until references are exhausted or the {@link #Loop_Limit} is
	reached.
<p>
	Map looping is used in conjuction with the {@link
	#LOOP_COUNTER_PARAMETER} in the default {@link #PVL_to_DB_GROUP}
	parameters. The value of the loop counter parameter is initialized to
	0 when {@link #PVL_Source() PVL_Source} mapping begins, and then
	incremented at the end of each mapping loop. By using the
	LOOP_COUNTER_PARAMETER in the {@link #Map(String) Map} a sequence of
	parameters from the PVL source can be mapped to the Database. For
	example, the Map entry:
<p>
	<blockquote><pre>
	Group = catalog.table
	&nbsp;&nbsp;field_1 = parameter@${PVL_to_DB/Loop_Counter}
	&nbsp;&nbsp;field_2 = array[${PVL_to_DB/Loop_Counter}]
	&nbsp;&nbsp;...
	End_Group
	</blockquote></pre>
<p>
	will insert a record in catalog.table with field_1 set to the value
	of the Nth parameter of that name and field_2 set to the Nth value
	of the array parameter, where N is the current value of the loop
	counter (starting with 0).
<p>
	<b>Note</b>: Map looping is disabled by default.
<p>
	@param	enable	true if map looping is to be enabled; false
		otherwise.
	@return	This PVL_to_DB object.
*/
public PVL_to_DB Loop
	(
	boolean	enable
	)
{
Loop = enable;
return this;
}

/**	Tests if map looping is enabled.
<p>
	@return	true if map looping is enabled; false otherwise.
	@see	#Loop(boolean)
*/
public boolean Loop ()
{return Loop;}

/*==============================================================================
	Configuration
*/
/**	Update the configuration.
<p>
	A {@link #PVL_to_DB_GROUP PVL_to_DB} Group, if not present, is added
	to the Configuration of the Database. In this parameter group the
	{@link #HOSTNAME_PARAMETER} is set to the name of the host system,
	and the {@link #LOOP_COUNTER_PARAMETER} is initialized to 0.
<p>
	If the {@link Conductor#UNRESOLVED_REFERENCE_PARAMETER} is present
	in the Database Configuration, its value is used to set the {@link
	Reference_Resolver#Default_Value(String) default value} for
	unresolved references. Otherwise the {@link #RESOLVER_DEFAULT_VALUE}
	is used.
<p>
	@throws	Configuration_Exception	If there is a problem accessing
		the configuration.
*/
private void Configure ()
	throws Configuration_Exception
{ 
if ((DEBUG & DEBUG_CONFIGURATION) != 0)
	System.out.println
		(">>> PVL_to_DB.Configure");
Configuration
	configuration = The_Database.Configuration ();
if ((DEBUG & DEBUG_CONFIGURATION) != 0)
	System.out.println
		("    The_Database.Configuration -" + NL
		+ configuration.Description () + NL);

//	Default value for an unresolved nested reference.
String
	default_value = configuration.Get_One
		(Conductor.UNRESOLVED_REFERENCE_PARAMETER);
if (default_value == null)
	default_value = RESOLVER_DEFAULT_VALUE;
else if (default_value.toUpperCase ().startsWith
		(Conductor.UNRESOLVED_REFERENCE_THROWS))
	default_value = null;
Default_Value (default_value);

//	Throw an exception on an unresolved source reference.
Unresolved_Source_Reference_Throws = configuration.Enabled
	(UNRESOLVED_SOURCE_REFERENCE_PARAMETER, UNRESOLVED_SOURCE_REFERENCE_THROWS);

/*	Initialize the PVL_to_DB_Group
	adding any PVL_to_DB group parameters from the Database Configuration.
*/
Default_Parameters (PVL_to_DB_Group (configuration));
if ((DEBUG & DEBUG_CONFIGURATION) != 0)
	System.out.println
		("    Effective configuration -" + NL
		+ configuration.Description () + NL
		+"<<< PVL_to_DB.Configure");
}

/*------------------------------------------------------------------------------
	Reference Resolver
*/
/**	Sets the {@link Reference_Resolver#Match_Mode(int) match mode} used
	to resolve both nested configuration parameter references and {@link
	#Map(String) Map} parameter source references from the {@link
	#PVL_Source() PVL source}.
<p>
	@param	match_mode	One of the {@link Reference_Resolver#CASE_SENSITIVE},
		{@link Reference_Resolver#CASE_INSENSITIVE}, or
		{@link Reference_Resolver#PATTERN_MATCH} values.
	@return	This PVL_to_DB object.
	@throws	IllegalArgumentException	If the match mode is not a
		valid value.
*/
public PVL_to_DB Match_Mode
	(
	int		match_mode
	)
	throws IllegalArgumentException
{
The_Resolver.Match_Mode (match_mode);
return this;
}

/**	Gets the parameter match mode.
<p>
	@return The current parameter match mode code.
	@see	#Match_Mode(int)
*/
public int Match_Mode ()
{return The_Resolver.Match_Mode ();}

/**	Sets the default value for unresolved nested references.
<p>
	<b>Note</b>: The default value may be set by using an {@link
	Conductor#UNRESOLVED_REFERENCE_PARAMETER Unresolved_Reference}
	parameter in the Configuration of the Database provided when the
	PVL_to_DB object is constructed. If the value of this parameter
	starts with {@link Conductor#UNRESOLVED_REFERENCE_THROWS "THROW"}
	(case insensitive) then unresolved nested references will throw an
	Unresolved_Reference exception.
<p>
	<b>N.B.</b>: The default value only applies to unresolved nested
	configuration parameter references. Source parameter (non-nested)
	references never have a default value.
<p>
	@param	default_value	The default String value for unresolved
		nested references. If null, unresolved references will throw an
		Unresolved_Reference expception.
	@return	This PVL_to_DB object.
	@see	Reference_Resolver#Default_Value(String)
*/
public PVL_to_DB Default_Value
	(
	String	default_value
	)
{
The_Resolver.Default_Value (default_value);
return this;
}

/**	Gets the default that will be used when a nested configuration
	parameter reference can not be resolved.
<p>
	@return	The String to use as the default. This is null if
		Unresolved_Reference will be thrown when a reference can not be
		resolved.
	@see	#Default_Value(String)
*/
public String Default_Value ()
{return The_Resolver.Default_Value ();}

/**	Force an unresolved source reference to throw an
	Unresolved_Reference exception.
<p>
	Unless the {@link #Configure() configuration} {@link
	#UNRESOLVED_SOURCE_REFERENCE_PARAMETER} or the {@link
	#UNRESOLVED_SOURCE_REFERENCE_THROWS} default value is true, a source
	reference that is not {@link #PVL_Source() resolved} will result in
	the corresponding database field being ignored. Otherwise an
	unresolved source reference will result in an Unresolved_Reference
	exception being thrown.
<p>
	<b>N.B.</b>: If {@link #No_Op(boolean) no-op} mode is enabled an
	exception will not be thrown in any case.
<p>
	@param	enabled	If true an unresolved source reference will result in
		an Unresolved_Reference exception being thrown. If false an
		unresolved source reference will result in the mapped database
		field being ignored.
	@return	This PVL_to_DB object.
*/
public PVL_to_DB Unresolved_Source_Reference_Throws
	(
	boolean	enabled
	)
{
Unresolved_Source_Reference_Throws = enabled;
return this;
}

/**	Test whether an unresolved source reference will throw an
	Unresolved_Reference exception.
<p>
	@return	true if an unresolved source reference will result in
		an Unresolved_Reference exception being thrown. false if an
		unresolved source reference will result in the mapped database
		field being ignored.
	@see	#Unresolved_Source_Reference_Throws(boolean)
*/
public boolean Unresolved_Source_Reference_Throws ()
{return Unresolved_Source_Reference_Throws;}

/**	Gets the Reference_Resolver being used to resolve all references.
<p>
	<b>Warning</b>: Modifying the Reference_Resolver could produce
	unexpected source mapping results.

	@return	The Reference_Resolver being used.
*/
public Reference_Resolver Resolver ()
{return The_Resolver;}

/*==============================================================================
	Source Parameters
*/
/**	Gets the current set of source parameters.
<p>
	@return	The Aggregate Parameter containing the parameters for
		resolving source parameter references. <b>N.B.</b>: After {@link
		#PVL_Source() source mapping} the source parameters include the
		{@link #Default_Parameters() default parameters} in a {@link
		#PVL_to_DB_GROUP} Aggregate.
	@see	#Source_Parameters(Parameter)
*/
public Parameter Source_Parameters ()
{return Source_Parameters;}

/**	Sets the source parameter(s) to be mapped.
<p>
	If the source is an Assignment a new Aggregate named {@link
	#DEFAULT_SOURCE_PARAMETERS_NAME} is created to contain it.
<p>
	The {@link #Default_Parameters(Parameter) default parameters} are updated:
	{@link #SOURCE_NAME_PARAMETER} is set to the name of the source
	Aggregate and
	{@link #SOURCE_PATHNAME_PARAMETER},
	{@link #SOURCE_DIRECTORY_PARAMETER},
	{@link #SOURCE_FILENAME_PARAMETER},
	{@link #SOURCE_FILENAME_ROOT_PARAMETER}, and
	{@link #SOURCE_FILENAME_EXTENSION_PARAMETER}
	are set to the corresponding components of the source name taken to
	be a file pathname. If, however, the name does not refer to an
	existing file, then these parameters will all be given the empty
	string as their value.
<p>
	@param	source	The Parameter source. If null, the current source
		is unchanged.
	@return	This PVL_to_DB object.
	@throws	Configuration_Exception	If there was a problem updating
		the default parameters from the source name.
	@see	#Default_Parameters()
	@see	Reference_Resolver#Parameters(Parameter)
*/
public PVL_to_DB Source_Parameters
	(
	Parameter	source
	)
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_SOURCE) != 0)
	System.out.println
		(">-< PLV_to_DB.Source_Parameters (Parameter)");
if (source != null)
	{
	if (source.Is_Aggregate ())
		Source_Parameters = source;
	else
		{
		Source_Parameters = new Parameter (DEFAULT_SOURCE_PARAMETERS_NAME)
			.Classification (Parameter.GROUP);
		try {Source_Parameters.Add (source);}
		catch (PVL_Exception exception) {/* Known to be an Aggregate */}
		}

	//	Set the current source in the PVL_to_DB Group.
	Update_Source_Name_Parameters ();
	}
return this;
}

/**	Sets the source parameter(s) to be mapped.

	@param	source	A PVL source String. This may be a URL, local
		pathname, or a system resource. If null, nothing is done.
	@throws	PVL_Exception	If there was a problem reading from or
		parsing the source stream.
	@throws	Configuration_Exception	If there was a problem updating
		the default parameters from the source name.
	@see	Streams#Get_Stream(String)
	@see	#Source_Parameters(Parameter)
*/
public PVL_to_DB Source_Parameters
	(
	String	source
	)
	throws PVL_Exception, Configuration_Exception
{
if ((DEBUG & DEBUG_SOURCE) != 0)
	System.out.println
		(">-< PLV_to_DB.Source_Parameters: " + source);
return Source_Parameters (Assemble_PVL (source));
}

/*------------------------------------------------------------------------------
	Default Parameters
*/
/**	Finds a PVL_to_DB Group in a Parameter list.
<P>
	@param	parameters	The Aggregate Parameter to search.
	@return	The top level {@link #PVL_to_DB_GROUP} Aggregate, or null if
		not found. The first group found is returned.
*/
public static Parameter PVL_to_DB_Group
	(
	Parameter	parameters
	)
{
if (parameters == null ||
	! parameters.Is_Aggregate () ||
	parameters.List_Size () == 0)
	return null;
Parameter
	parameter = null;
while ((parameter = parameters.Find
			(Configuration.Absolute_Pathname (PVL_to_DB_GROUP),
				false, parameter)) != null &&
		! parameter.Is_Aggregate ());
if (parameter != null &&
	! parameter.Is_Aggregate ())
	parameter = null;
return parameter;
}

/**	Removes all PVL_to_DB Groups in a Parameter list.
<P>
	@param	parameters	The Aggregate Parameter to search.
	@return	The last top level {@link #PVL_to_DB_GROUP} Aggregate
		that was removed; null otherwise.
*/
public static Parameter Remove_PVL_to_DB_Group
	(
	Parameter	parameters
	)
{
if (parameters == null ||
	! parameters.Is_Aggregate () ||
	parameters.List_Size () == 0)
	return null;
Parameter
	removed = null;
Parameter
	parameter = null;
while ((parameter = PVL_to_DB_Group (parameters)) != null)
	removed = parameters.Remove (parameter);
return removed;
}

/**	Adds new parameters to the default PVL_to_DB Group.
<P>
	The PVL_to_DB Group of parameters is used as defaults when resolving
	source parameter references in the map that are not resolved by the
	{@link #Source_Parameters(Parameter) Source Parameters}. Just before
	the {@link #PVL_Source() source mapping operation} begins the
	default parameters are appended to the source parameters which
	ensures that the default parameters will be present for source
	parameter resolution, but the default parameters will not override
	any source parameters having the same name.
<P>
	If the parameter is an Assignment it is added to the {@link
	#PVL_to_DB_GROUP PVL_to_DB Group} of default parameters. If the
	Parameter is an Aggregate and it contains a PVL_to_DB Group, the
	parameters in that group are added to the PVL_to_DB Group; otherwise
	all the Parameters of the Aggregate are added to the PVL_to_DB
	Group. The PVL_to_DB Group is Coalesced with its {@link
	Configuration#Duplicate_Parameter_Action(int) duplicate parameter
	action} set so that newly added duplicate parameter names will take
	precedence.
<p>
	If the Parameter is null the PVL_to_DB Group is reset with the
	default parameters, which are always included:
<dl>
	<dt>{@link #HOSTNAME_PARAMETER}
		<dd>The name of the host system.
	<dt>{@link #CWD_PARAMETER}
		<dd>The pathname to the current working directory.
	<dt>{@link #SOURCE_NAME_PARAMETER}
		<dd>The current {@link #PVL_Source() PVL_Source} name.
	<dt>{@link #SOURCE_PATHNAME_PARAMETER}
		<dd>The absolute pathname of the current {@link #PVL_Source()
		PVL_Source}. This will be empty if the file does not exist.
	<dt>{@link #SOURCE_DIRECTORY_PARAMETER}
		<dd>The directory pathname of the SOURCE_FILENAME_PARAMETER.
		This will be empty if the file does not exist.
	<dt>{@link #SOURCE_FILENAME_PARAMETER}
		<dd>The filename portion of the SOURCE_PATHNAME_PARAMETER.
		This will be empty if the file does not exist.
	<dt>{@link #SOURCE_FILENAME_ROOT_PARAMETER}
		<dd>The SOURCE_FILENAME_PARAMETER value with the extension removed.
		This will be empty if the file does not exist.
	<dt>{@link #SOURCE_FILENAME_EXTENSION_PARAMETER}
		<dd>The extension portion of the SOURCE_FILENAME_PARAMETER.
		This will be empty if the filename has no extension following a
		period ('.') character, or if the file does not exist.
	<dt>{@link #LOOP_COUNTER_PARAMETER}
		<dd>The {@link #Loop() Loop} counter; initialized to 0.
</dl><p>
	@param	parameter	The parameter(s) to be added to the PVL_to_DB
		default source parameters group.
	@return	This PVL_to_DB object.
	@throws	Configuration_Exception	If there is a problem adding a
		new parameter.
	@see	#Default_Parameter(String, Object)
*/
public PVL_to_DB Default_Parameters
	(
	Parameter	parameter
	)
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_CONFIGURATION) != 0)
	System.out.println
		(">>> PVL_to_DB.Default_Parameters");
if (parameter == null)
	{
	if (PVL_to_DB_Group == null)
		{
		//	Should only happen at startup.
		if ((DEBUG & DEBUG_CONFIGURATION) != 0)
			System.out.println
				("    Constructing empty " + PVL_to_DB_GROUP
					+ " parameters group.");
		PVL_to_DB_Group = new Configuration ((Parameter)null);
		PVL_to_DB_Group.Name (PVL_to_DB_GROUP);
		PVL_to_DB_Group.Duplicate_Parameter_Action
			(Configuration.PREFER_LAST_PARAMETER);
		}
	else
		{
		//	Remove all parameters.
		if ((DEBUG & DEBUG_CONFIGURATION) != 0)
			System.out.println
				("    Resetting " + PVL_to_DB_GROUP + " parameters group.");
		PVL_to_DB_Group.Remove_All ();
		}
	try
		{
		//	Add the default parameters.
		if ((DEBUG & DEBUG_CONFIGURATION) != 0)
			System.out.println
				("    Adding required default parameters.");
		Default_Parameter (HOSTNAME_PARAMETER, Host.Hostname ());
		Default_Parameter (CWD_PARAMETER, System.getProperty ("user.dir"));
		Update_Source_Name_Parameters ();
		Default_Parameter (LOOP_COUNTER_PARAMETER, new Integer (0));
		}
	catch (Configuration_Exception exception)
		{
		throw new Configuration_Exception (Error_Message
			("Unable to set the default parameters in the "
				+ PVL_to_DB_GROUP + " group." + NL
			+ exception.getMessage ()));
		}
	}
else
	{
	if (PVL_to_DB_Group == null)
		//	Provide the default parameters.
		Default_Parameters (null);
	Parameter
		group = parameter;
	try
		{
		if (parameter.Is_Aggregate ())
			{
			if ((DEBUG & DEBUG_CONFIGURATION) != 0)
				System.out.println
					("    Adding optional default parameters.");
			if ((group = PVL_to_DB_Group (parameter)) == null)
				{
				//	Add all parameters.
				if ((DEBUG & DEBUG_CONFIGURATION) != 0)
					System.out.println
						("    Using all parameters -");
				group = parameter;
				}
			else if ((DEBUG & DEBUG_CONFIGURATION) != 0)
				System.out.println
					("    Only using " + PVL_to_DB_Group + " group -");

			//	Copy the Parameter list into the internal PVL_to_DB Group.
			if ((DEBUG & DEBUG_CONFIGURATION) != 0)
				System.out.println (group.Description ());
			group = new Parameter (group);
			PVL_to_DB_Group.Add (group.List ());
			}
		else
			{
			if ((DEBUG & DEBUG_CONFIGURATION) != 0)
				System.out.println
					("    Adding optional default parameter -" + NL
					+ parameter.Description ());
			PVL_to_DB_Group.Add (parameter);
			}

		//	Coalesce the parameters.
		PVL_to_DB_Group.Coalesce ();
		}
	catch (PVL_Exception exception)
		{
		throw new Configuration_Exception (Error_Message
			("Unable to add default parameters from "
				+ group.Path_Name () + NL
			+ exception.getMessage ()));
		}
	}
if ((DEBUG & DEBUG_CONFIGURATION) != 0)
	System.out.println
		(PVL_to_DB_Group.Description ()
		+"<<< PVL_to_DB.Default_Parameters");
return this;
}

/**	Gets the default PVL_to_DB Group of parameters.
<p>
	@return	The PVL_to_DB Aggregate Parameter.
*/
public Parameter Default_Parameters ()
{return PVL_to_DB_Group;}

/**	Adds a new parameter to the default PVL_to_DB Group.
<p>
	@param	name	The name of the parameter to be set.
	@param	value	An Object to use for the parameter's value.
	@return	true if an existing parameter by the same name was
		replaced; false if the parameter is being set for the
		first time.
	@throws	Configuration_Exception	If there is a problem adding a
		new parameter.
	@see	#Default_Parameters(Parameter)
*/
public boolean Default_Parameter
	(
	String	name,
	Object	value
	)
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_CONFIGURATION) != 0)
	System.out.println
		(">-< PVL_to_DB.Default_Parameter: " + name + " = " + value);
try {return PVL_to_DB_Group.Set (name, value);}
catch (Configuration_Exception exception)
	{
	throw new Configuration_Exception (Error_Message
		("Unable to set default parameter: " + name + " = " + value + NL
		+ exception.getMessage ()));
	}
}

/**	Updates the SOURCE_XXX parameters in the PVL_to_DB group.
<p>
	The {@link #SOURCE_NAME_PARAMETER} is set to the name of the {@link
	#Source_Parameters()} Aggregate. If there is none (it is null)
	then the name will be the empty string.
<p>
	The {@link #SOURCE_PATHNAME_PARAMETER},
	{@link #SOURCE_DIRECTORY_PARAMETER},
	{@link #SOURCE_FILENAME_PARAMETER},
	{@link #SOURCE_FILENAME_ROOT_PARAMETER}, and
	{@link #SOURCE_FILENAME_EXTENSION_PARAMETER} Parameters are set to
	the corresponding components of the {@link #Source_Parameters()}
	Aggregate name taken to be a pathname (absolute or relative). If,
	however, there is no Aggregate or its name does not refer to an
	existing file, then these parameters will all be given the empty
	string as their value.

	The (@link LOOP_COUNTER_PARAMETER} is reset to zero.
*/
private void Update_Source_Name_Parameters ()
	throws Configuration_Exception
{
String
	source_name = "";
if (Source_Parameters != null)
	source_name = Source_Parameters.Name ();
Default_Parameter (SOURCE_NAME_PARAMETER, source_name);

File
	file = new File (source_name);
if (file.exists ())
	{
	if (! file.isAbsolute ())
		file = new File (file.getAbsolutePath ());
	Default_Parameter (SOURCE_PATHNAME_PARAMETER,
		file.getAbsolutePath ());
	Default_Parameter (SOURCE_DIRECTORY_PARAMETER,
		file.getParent ());
	source_name = file.getName ();
	Default_Parameter (SOURCE_FILENAME_PARAMETER,
		source_name);
	int
		index = source_name.lastIndexOf ('.');
	if (index < 0)
		index = source_name.length ();
	Default_Parameter (SOURCE_FILENAME_ROOT_PARAMETER,
		source_name.substring (0, index));
	if (index < source_name.length ())
		index++;
	Default_Parameter (SOURCE_FILENAME_EXTENSION_PARAMETER,
		source_name.substring (index));
	}
else
	{
	Default_Parameter (SOURCE_PATHNAME_PARAMETER, "");
	Default_Parameter (SOURCE_DIRECTORY_PARAMETER, "");
	Default_Parameter (SOURCE_FILENAME_PARAMETER, "");
	Default_Parameter (SOURCE_FILENAME_ROOT_PARAMETER, "");
	Default_Parameter (SOURCE_FILENAME_EXTENSION_PARAMETER, "");
	}
}

/**	Updates the {@link #LOOP_COUNTER_PARAMETER} in the
	Source_Parameters.
<p>
	@param	count	The new loop count value.
*/
private void Update_Loop_Counter_Parameter
	(
	int		count
	)
{
if (Source_Parameters != null)
	{
	Parameter
		parameter = Source_Parameters.Find
			(Configuration.Absolute_Pathname
				(PVL_to_DB_GROUP, LOOP_COUNTER_PARAMETER));
	if (parameter != null)
		{
		try {parameter.Value (count);}
		catch (PVL_Exception exception) {/* Shouldn't happen */}
		}
	}
}

/**	Gets the value of the loop counter.
<p>
	This will be the number of times the last {@link #PVL_Source()
	PVL_Source} was successfully mapped to the Database.
*/
public int Loop_Count ()
{
if (Source_Parameters != null)
	{
	Parameter
		parameter = Source_Parameters.Find
			(Configuration.Absolute_Pathname
				(PVL_to_DB_GROUP, LOOP_COUNTER_PARAMETER));
	if (parameter != null)
		{
		try {return (int)parameter.Value ().long_Data ();}
		catch (PVL_Exception exception) {/* Shouldn't happen. */}
		}
	}
return 0;
}

/*==============================================================================
	Map
*/
/**	Assembles a map of PVL source parameter references assigned to
	Database field references.
<p>
	A PVL_to_DB map is used to assign Parameter (or constant) values to
	Database fields. It provides the mapping from resolved source
	reference values to resolved database field references where the
	values are deposited. <b>Note</b>: No map Parameter Value may be an
	Array; there must be only one Parameter Value for each Database field
	reference. However, this Value may contain multiple source references
	that are combined to produce a single field value.
<p>
	Syntactically, the map is a Aggregate of PVL Parameters in which the
	name of each Assignment Parameter is a Database field reference and
	the Value is a source reference:
<p>
	<blockquote>
	<b><i>Database_field_reference</i> <b>=</b> <i>Source_reference</i></b>
	</blockquote>
<p>
<h4>
	Nested references -
</h4><p>
	Any reference may contain {@link Reference_Resolver nested
	references}. All nested references are recursively resolved before
	the reference that contains them. A nested reference may be either a
	configuration parameter reference or a database field reference. All
	nested references are enclosed in curly braces ('{' and '}'), with
	configuration parameter references being distinguished from database
	field references by the former having a leading dollar sign ('$')
	before the opening curly brace. All nested references will either be
	resolved to a value or the {@link #Default_Value(String)
	Default_Value} will be used (if the default value is null an
	Unresolved_Reference exception will be thrown).
<p>
	<b>N.B.</b>: Nested parameter references are resolved against the
	configuration parameters, not the source parameters. Only nested
	references are enclosed in curly braces. The final database field
	and parameter source references used to accomplish the value
	assignment must not be enclosed.
<p>
<h4>
	Database field reference -
</h4><p>
	<blockquote>
	<i>Catalog</i><b>.</b><i>Table</i><b>.</b><i>Field</i><b>:</b><i>Key</i>
	</blockquote>
<p>
	The components preceding the colon form a fully qualified field
	name, and the component following the colon is a match (SQL WHERE)
	criteria used to identify a specific record from the database.
<h5>
	Insert parameters
</h5><p>
	Map parameters that have a database reference without a key will
	insert a new record in the field of the table specified by the
	reference.
<h5>
	Update parameters
</h5><p>
	Map parameters that have a database reference with a key are used
	to specify existing database fields to be updated with a new value.
	The key is the criteria that identifies the specific record of the
	table that is to be updated. The key is required to uniquely
	identify a single record.
<h5>
	Update or Insert
</h5><p>
	A database reference may be flagged with an {@link
	#UPDATE_OR_INSERT_MARKER} prefix character that will cause an update
	to be tried if the reference includes a key. If the update fails then
	an insert will be performed. If the reference does not include a key
	then only the insert is done. The special prefix marker is, of
	course, removed from the database reference before it is used.
<h4>
	Source reference -
</h4>
	A source reference is the Value of a map's Assignment Parameter. It
	is the expression that must be resolved to a value that is assigned
	to the database field. Ultimately each source reference must resolve
	to a single string value, which may be the text representation of a
	numeric value. However, a source reference is evaluated as
	a sequence of tokens that are individually resolved to string values
	that are concatenated to form the final source reference resolution.
	See the description of {@link #Resolve_Source_References(String)
	source references resolving} for the details of the source reference
	expression syntax.
<h5>
	Parameter source reference -
</h5>
	The basic source reference names a parameter in the source file which
	has an assignment value that is used to satisfy the reference. The
	syntax of a parameter source reference is the sames as the syntax for
	a nested parameter reference, but without the enclosing "${" and "}"
	characters:
<p>
	<blockquote>
	<i>Pathname</i><b>@</b><i>Sequence</i><b>[</b><i>Index</i><b>]</b>...
	</blockquote>
<p>
	The <I>Pathname</I> is a simple, relative or absolute Parameter
	pathname used to {@link PIRL.PVL.Parameter#Find(String) Find} the
	parameter in the {@link #PVL_Source() PVL source}. How the
	pathname matching is done - whether case sensitive or using a
	regular expression pattern - is controlled by by the {@link
	#Match_Mode(int) match mode}. The <I>Sequence</I>, with its
	preceeding "at" ('@') delimiter , is optional; if present it
	indicates which of a multiple sequence parameters that match the
	Pathname to select (the first parameter is 0, which is the
	default). The <I>Index</I>, with its enclosing square bracket ('['
	and ']') characters, is also optional; if present it indicates
	which element of an Array Value to select (the first element is 0,
	which is the default). Multiple Array Value element specifiers may
	be provided to select from multidimensional Array Values. Both the
	Sequence and Index specifiers may contain nested references. They
	may also be mathematical expressions that evaluate to any numeric
	value, which is truncated to an integer for selecting Array Value
	elements. The typical parameter source reference is simply a
	name of a source parameter.
<p>
	<b>N.B.</b>: Source parameter references are resolved against the
	source parameters, not the configuration parameters.
<h4>
	Groups
</h4><p>
	Map parameters that are contained within an Aggregate (Group)
	Parameter define the field values for a single record in the
	Database. The name of the Group is a Database {@link
	Database#Table_Reference(String, String) table reference} (in
	<i>Catalog</i>.<i>Table</i> notation) that specifies the table where
	the record will located; that is it is a database field reference
	without the <i>Field</i> portion. Each parameter within the Group
	specifies a field name and its source reference value, therefore its
	database reference can contain just the field name without the
	catalog and table name portions (if these are present, however, they
	must be the same as the table reference in the Group name); the table
	reference portion is implicitly the name of the Group.
<p>
	<b>N.B.</b>: As a special case, if the Group name contains nested
	references that can not be resolved against the configuration
	parameters an attempt will be made to resolve against both the
	configuration parameters and the current source parameters, in that
	order. This is a deprecated feature that could result in confusion
	of the correct parameter to resolve a nested reference. Retry can
	be disabled by setting Retry_Group_with_Source to false.
<p>
	The field names and values of a Group are all collected together to
	produce a single operation on the Database. Groups, therefore, are
	particularly useful for specifying all the fields of a single new
	record to be inserted into a table; the alternative is to insert a
	single field of a new record and then apply updates to this new
	record to provide additional field values. If the database reference
	of the Group name includes a key, then a single Database update
	operation will be done. <b>N.B.</b>: Depending on the particular
	database server, this could result in only some of the updates being
	completed if there is an error in one of the map references within
	the Group.
<h4>
	Quotes and Escapes
</h4><p>
	Because source references may be complex expressions and are likely
	to contain special PVL syntax characters (e.g. '{', '}', '=', quotes
	and spaces), the entire Value portion of a Map Parameter should be
	quoted (with either single or double quote characters) to prevent the
	misinterpretation of the special characters. <b>This is a common
	source of syntax errors.</b>
<p>
	When characters that would otherwise be interpreted as nested
	reference enclosure characters, or other special parse control
	characters (such as quotes and parentheses), need to be passed
	through as the value of the reference preceed these characters with a
	backslash ('\') which will escape the usual interpretation of the
	following character and remove the backslash. The backslash character
	itself may be escaped in this way.
<p>
	@param	source	The name of the PVL source (URL or pathname)
		containing the PVL that defines the map.
	@return	A PVL Parameter Aggregate containing the mapping structure.
	@throws	PVL_Exception	If there is a problem parsing the PVL.
*/
public PVL_to_DB Map
	(
	String	source
	)
	throws PVL_Exception
{
The_Map = Assemble_PVL (source);
return this;
}

/**	Sets the map parameters.
<p>
	@param	map	The map Parameter Aggregate.
	@return	This PVL_to_DB object.
	@throws	IllegalArgumentException	If the map is not an Aggregate
		Parameter.
*/
public PVL_to_DB Map
	(
	Parameter	map
	)
	throws IllegalArgumentException
{
if (map != null &&
	! map.Is_Aggregate ())
	throw new IllegalArgumentException (Error_Message (
		"The Map is not an Aggregate - " + map.Name ()));
The_Map = map;
return this;
}

/**	Gets the map Parameters.
<p>
	@return	The map Parameter Aggregate.
	@see	#Map(Parameter)
*/
public Parameter Map ()
{return The_Map;}

/*==============================================================================
	Mappers
*/
/**	The current source of PVL parameters is mapped into the Database.
<p>
	The {@link #Loop_Count() loop counter} is first reset to zero.
<p>
	If there are no source parameters or map, then nothing is done.
<p>
	Any top level {@link #PVL_to_DB_GROUP} Aggregates are removed (as
	would occur if a source were reprocessed for any reason). Then the
	{@link #Default_Parameters(Parameter) default parameters} are
	appended to the {@link #Source_Parameters(Parameter) source
	parameters}.
<p>
	The map processing loop is entered. The loop counter is incremented
	at the successful completion of this loop, but the loop number (the
	first loop is number 1) is reported in verbose mode. In verbose
	mode all mapping operations - insert or update - and their values -
	both before and after reference resolution - will be reported. At
	the end of each map processing loop the number of updates effected
	on the database and records that were affected is listed, and after
	map processing completes the totals of these values will be listed.
	If map {@link #Loop(boolean) looping} has been enabled the map
	processing loop will be run until the loop counter reaches the
	{@link #Loop_Limit} or no source parameters can be resolved during
	a processing loop. The map will be processed only once if looping
	is disabled.
<p>
	Each map parameter is processed in the order it occurs in the map.
	A database reference is obtained from each parameter name and
	resolved of all nested references. From this a table reference and
	key is identified. Each database reference must contain a table
	reference, but the key is only present for update (not insert)
	operations. For Assignments the parameter value is resolved to a
	source reference (the details are provided in the description of
	the {@link #Map(String) map} syntax). For Aggregates all of the
	Assignment parameters it contains have their the database field
	references and resolved parameter references collected together
	before a database operation will be performed. If a source
	parameter reference can not be resolved the parameter is ignored.
	<b>N.B.</b>: Nested parameter references may be resolved by a
	{@link #Default_Value(String) default value} or throw an
	Unresolved_Reference exception if no default has been specified,
	but this only applies to nested references; after resolving nested
	references the result is still treated as a parameter reference but
	without a default (or exception). If at least one source parameter
	reference has been resolved, and {@link #No_Op(boolean) no-op} mode
	has not been enabled, then the Database is updated.
<p>
	@return	The total number of database record updates/inserts.
	@throws	ParseException	If a reference could not be resolved due
		to invalid syntax.
	@throws	Database_Exception	If there is a problem accessing
		the database.
	@throws	Invalid_Map_Syntax	If the map parameters contain
		a invalid syntax; such as the lack of a valid table
		reference, an assignment parameter without a field name,
		an aggregate contained inside another aggregate, or an
		aggregate parameter entry with a table reference that does not
		match it's conataining aggregate.
	@throws	Unresolved_Reference	If a nested reference could not be
		resolved and there was no {@link #Default_Value(String)
		Default_Value}.
	@see	Update_DB#Update_Database(String, String, Vector, Vector)
*/
public int PVL_Source ()
	throws
		ParseException,
		Database_Exception,
		Invalid_Map_Syntax,
		Unresolved_Reference
{
if ((DEBUG & DEBUG_SOURCE) != 0)
	{
	System.out.println
		(">>> PLV_to_DB.PVL_Source");
	Verbose = true;
	}
if (Source_Parameters == null ||
	The_Map == null)
	{
	if ((DEBUG & DEBUG_SOURCE) != 0)
		System.out.println ("<<< PVL_to_DB.PVL_Source: 0 (no "
			+ ((Source_Parameters == null) ? "source" : "map") + ")");
	return 0;
	}

//	Remove any existing PVL_to_DB groups.
Remove_PVL_to_DB_Group (Source_Parameters);

//	Append the default PVL_to_DB group parameters to the source parameters.
try {Source_Parameters.Add (PVL_to_DB_Group);}
catch (PVL_Exception exception) {/* Known to be an Aggregate */}

if (Verbose || No_Op)
	System.out.println ("PVL source: " + Source_Parameters.Name ());

if ((DEBUG & DEBUG_SOURCE) != 0)
	System.out.println
		("    PLV_to_DB.PVL_Source: PVL -" + NL
		+ Source_Parameters.Description ());
int
	loop = 0,			//	Map processing loop counter.
	loop_limit = (Loop ? Loop_Limit : 1),
	records = 0,		//	Record updates per loop (from Update_Database).
	updates = -1,		//	Database updates per loop.
	total_records = 0,	//	Total number of record updates.
	total_updates = 0;	//	Total number of updates.
Parameter
	map_parameter,
	parameter;
String
	database_reference,
	table_reference,
	table_name,
	field,
	value,
	key,
	comment;
boolean
	update_or_insert;
Vector
	fields = new Vector (),
	values = new Vector ();

//	Map processing:

for (loop = 0;
	updates != 0 && loop < loop_limit;
	loop++)
	{
	Update_Loop_Counter_Parameter (loop);

	if (loop_limit > 1 &&
		(Verbose || No_Op))
		System.out.println
			(NL
			+"*** Loop " + (loop + 1));

	updates = records = 0;
	Iterator
		map_parameters = The_Map.iterator ();
	while (map_parameters.hasNext ())
		{
		map_parameter = (Parameter)map_parameters.next ();

		//	Clear the vectors of field names and values to be updated.
		values.clear ();
		fields.clear ();

		if (Verbose || No_Op)
			{
			if (map_parameter.Is_Aggregate ())
				System.out.println ("=== Group  " + map_parameter.Name ());
			else
				{
				comment = map_parameter.Comments ();
				map_parameter.Comments (null);
				System.out.println ("--- " + map_parameter.Description ());
				map_parameter.Comments (comment);
				}
			}

		//	Database reference from parameter.
		database_reference =
			Database_Reference (map_parameter,
				map_parameter.Is_Aggregate () ? Source_Parameters : null);
		if (database_reference == null ||
			database_reference.length () == 0 ||
			database_reference.charAt (0) != UPDATE_OR_INSERT_MARKER)
			update_or_insert = false;
		else
			{
			database_reference = database_reference.substring (1);
			update_or_insert = true;
			}

		//	Table reference from database reference
		if ((table_reference = Table_Reference (database_reference)) == null)
			throw new Invalid_Map_Syntax (Error_Message
				("No table reference for map parameter database reference."),
				database_reference, map_parameter);

		//	Key from database reference.
		key = The_Resolver.Key (database_reference);

		if (map_parameter.Is_Aggregate ())
			{
			if (Verbose || No_Op)
				{
				if (key == null)
					System.out.println
						("=== Insert " + table_reference);
				else
					System.out.println
						("=== Update "
							+ (update_or_insert ? "or Insert " : "")
							+ table_reference + The_Resolver.KEY_MARKER + key);
				}

			//	Collect all of the Aggregate's field names and values.
			Iterator
				parameters = map_parameter.iterator ();
			while (parameters.hasNext ())
				{
				parameter = (Parameter)parameters.next ();

				if (Verbose || No_Op)
					{
					comment = parameter.Comments ();
					parameter.Comments (null);
					System.out.println ("==> " + parameter.Description ());
					parameter.Comments (comment);
					}

				//	Parameter reference.
				try {value = Resolve_Source_Reference (parameter);}
				catch (Database_Exception exception)
					{
					throw new Database_Exception
						(exception.getMessage () + NL
						+"Parameter group: " + map_parameter.Name ());
					}
				catch (ParseException exception)
					{
					throw new ParseException
						(exception.getMessage () + NL
						+"Parameter group: " + map_parameter.Name (),
						exception.getErrorOffset ());
					}
				catch (Unresolved_Reference exception)
					{
					throw new Unresolved_Reference
						(exception.getMessage () + NL
						+"Parameter group: " + map_parameter.Name ());
					}

				if (value == null)
					{
					if (Verbose || No_Op)
						System.out.println ("!!! No resolution.");
					//	Didn't resolve to a value.
					continue;
					}

				//	Database reference.
				database_reference = Database_Reference (parameter);

				//	Table reference from database reference
				table_name = Table_Reference (database_reference);

				if (table_name != null &&
					! The_Database.Matches (table_name, table_reference))
					throw new Invalid_Map_Syntax (Error_Message
						("Map aggregate table reference \""
							+ table_reference + '"' + NL
						+"does not match the map parameter database reference."),
						table_name, parameter);

				//	Field name.
				if ((field = The_Resolver.Field_Name (database_reference))
						== null)
					throw new Invalid_Map_Syntax (Error_Message
						("No field name in map parameter database reference."),
						database_reference, parameter);

				fields.add (field);
				values.add (value);

				if (Verbose || No_Op)
					System.out.println ("    " + field + " = " + value);
				}
			}
		else	//	Top level Assignment.
			{
			//	Parameter reference.
			value = Resolve_Source_Reference (map_parameter);
			if (value == null)
				{
				if (Verbose || No_Op)
					System.out.println ("!!! No resolution.");
				//	Didn't resolve to a value.
				continue;
				}

			//	Field name.
			if ((field = The_Resolver.Field_Name (database_reference))
					== null)
				throw new Invalid_Map_Syntax (Error_Message
					("No field name in map parameter database reference."),
					database_reference, map_parameter);

			fields.add (field);
			values.add (value);

			if (Verbose || No_Op)
				System.out.println ("--> "
					+ ((key == null) ? "Insert" : "Update") + ' '
					+ table_reference + The_Resolver.COMPONENTS_DELIMITER
					+ field + " = " + value);
			}

		if (! fields.isEmpty ())
			{
			if ((DEBUG & DEBUG_SOURCE) != 0)
				System.out.println
					("    PLV_to_DB.PVL_Source: Update_Database -" + NL
					+"    table_reference: " + table_reference + NL
					+"                key: " + key + NL
					+"    fields: " + fields + NL
					+"    values: " + values);
			if (! No_Op)
				{
				Exception
					exception = null;
				try {records +=
					Update_Database (table_reference, key, fields, values);}
				catch (Database_Exception except)
					{
					except = new Database_Exception (Error_Message
							(except.getMessage () + NL
							+"While mapping "
								+ (map_parameter.Is_Aggregate () ?
									("group " + map_parameter.Name ()) :
									(map_parameter.Description ()))));
					if (key == null ||
						! update_or_insert)
						throw except;
					exception = except;
					}
				catch (IllegalArgumentException except)
					{
					if (key == null ||
						! update_or_insert)
						throw except;
					exception = except;
					}
				if (exception != null)
					{
					//	Try to insert instead of update.
					try {records +=
						Update_Database (table_reference, null, fields, values);}
					catch (Database_Exception except)
						{
						throw new Database_Exception (Error_Message
							(except.getMessage () + NL
							+"While mapping "
								+ (map_parameter.Is_Aggregate () ?
									("group " + map_parameter.Name ()) :
									(map_parameter.Description ())) + NL
							+ NL
							+"During insert after record update failed -" + NL
							+ exception.getMessage ()));
						}
					}
				}
			++updates;
			}
		}
	if ((DEBUG & DEBUG_SOURCE) != 0)
		System.out.println
		("    ----------------------------------------------------------------" + NL);
	total_updates += updates;
	total_records += records;
	if (Verbose || No_Op)
		System.out.println
			(NL
			+"Updates effected: " + updates + NL
			+"Records affected: " + records);
	}

Update_Loop_Counter_Parameter (loop);
if (loop > 1 &&
	(Verbose || No_Op))
	System.out.println
		(NL
		+"Total updates effected: " + updates + NL
		+"Total records affected: " + records);
if ((DEBUG & DEBUG_SOURCE) != 0)
	System.out.println
		("<<< PVL_to_DB.PVL_Source: " + total_records);
return total_records;
}

/**	Names a source of PVL Parameters which is mapped into the Database.
<p>
	@param	source	The name of the PVL source (a local pathname or URL).
		If null, the previous PVL source is used. If there is no
		previous PVL source, nothing is done.
	@return	The total number of database record updates/inserts.
	@throws	PVL_Exception	If there is a problem parsing the PVL from
		the input file.
	@throws	Configuration_Exception	If there was a problem updating
		the default parameters from the source name.
	@throws	ParseException	If a reference could not be resolved due
		to invalid syntax.
	@throws	Database_Exception	If there is a problem accessing
		the database.
	@throws	Invalid_Map_Syntax	If the map parameters contain
		a invalid syntax; such as the lack of a valid table
		reference, an assignment parameter without a field name,
		an aggregate contained inside another aggregate, or an
		aggregate parameter entry with a table reference that does not
		match it's conataining aggregate.
	@throws	Unresolved_Reference	If a nested reference could not be
		resolved and there was no {@link #Default_Value(String)
		Default_Value}.
	@see	#Source_Parameters(String)
	@see	#PVL_Source()
*/
public int PVL_Source
	(
	String	source
	)
	throws
		PVL_Exception,
		Configuration_Exception,
		ParseException,
		Database_Exception,
		Invalid_Map_Syntax,
		Unresolved_Reference
{
//	Set the source of parameters (no change if null).
Source_Parameters (source);
//	Map the source parameters to the database.
return PVL_Source ();
}

/**	Supplies a source of PVL Parameters which is mapped into the Database.
<p>
	@return	The total number of database record updates/inserts.
	@throws	Configuration_Exception	If there was a problem updating
		the default parameters from the source name.
	@throws	ParseException	If a reference could not be resolved due
		to invalid syntax.
	@throws	Database_Exception	If there is a problem accessing
		the database.
	@throws	Invalid_Map_Syntax	If the map parameters contain
		a invalid syntax; such as the lack of a valid table
		reference, an assignment parameter without a field name,
		an aggregate contained inside another aggregate, or an
		aggregate parameter entry with a table reference that does not
		match it's conataining aggregate.
	@throws	Unresolved_Reference	If a nested reference could not be
		resolved and there was no {@link #Default_Value(String)
		Default_Value}.
	@see	#Source_Parameters(Parameter)
	@see	#PVL_Source()
*/
public int PVL_Source
	(
	Parameter	source
	)
	throws
		Configuration_Exception,
		ParseException,
		Database_Exception,
		Invalid_Map_Syntax,
		Unresolved_Reference
{
//	Set the source of parameters (no change if null).
Source_Parameters (source);
//	Map the source parameters to the database.
return PVL_Source ();
}

/*------------------------------------------------------------------------------
	Map reference resolvers
*/
/**	Gets a database reference from a map parameter.
<p>
	The name of the map parameter contains the database reference.
	All nested references in the name are resolved.
<p>
	If the initial attempt to resolve the map parameter name fails due
	to an Unresolved_Reference exception and retry parameters are
	provided the retry parameters will be temporarily bound to the
	reference resolver, after saving the previously bound parameters,
	and another attempt will be made to resolve the map parameter name.
	If this fails for any reason an Unresolved_Reference will be thrown.
	The original exception message will be used supplemented with the
	new exception message if the latter is different from the former.
	The parameters previously bound to the reference resolver will
	always be restored. <b>N.B.</b>: This is a deprecated workaround
	to provide backwards compatibility with a previous logic flaw and
	is expected to be removed in a future release.
<p>
	@param	map_parameter	The map parameter which has a database
		reference as its name.
	@param	retry_parameters	Parameters to use for resolver retry.
		If null no retry is attempted.
	@return	The resolved database reference String.
	@throws	ParseException	if a nested reference could not be resolved due
		to invalid syntax.
	@throws	Database_Exception	if there was a problem accessing
		the database to resolve a nested database reference.
	@throws	Unresolved_Reference	if a nested reference could not be
		resolved and there was no {@link #Default_Value(String)
		Default_Value}.
*/
private String Database_Reference
	(
	Parameter	map_parameter,
	Parameter	retry_parameters
	)
	throws
		ParseException,
		Database_Exception,
		Unresolved_Reference
{
if ((DEBUG & (DEBUG_FIELD_REFERENCE | DEBUG_SOURCE)) != 0)
	System.out.println
		(">>> PVL_to_DB.Database_Reference: " + map_parameter.Name ());
String
	database_reference = null;
try {database_reference = The_Resolver.Resolve (map_parameter.Name ());}
catch (ParseException exception)
	{
	throw new ParseException (Error_Message
		("For the database reference of map entry" + NL
		+ (map_parameter.Is_Aggregate () ?
			(map_parameter.Name () + NL) :
			map_parameter.Description ())
		+ exception.getMessage ()),
		exception.getErrorOffset ());
	}
catch (Database_Exception exception)
	{
	throw new Database_Exception (Error_Message
		("For the database reference of map entry" + NL
		+ (map_parameter.Is_Aggregate () ?
			(map_parameter.Name () + NL) :
			map_parameter.Description ())
		+ exception.getMessage ()));
	}
catch (Unresolved_Reference exception)
	{
	String
		message = exception.getMessage ();
	Unresolved_Reference
		unresolved_reference =
			new Unresolved_Reference (Error_Message
				("For the database reference of map entry" + NL
				+ (map_parameter.Is_Aggregate () ?
					(map_parameter.Name () + NL) :
					map_parameter.Description ())
				+ message),
				map_parameter.Name ());

	if (Retry_Group_with_Source &&
		retry_parameters != null)
		{
		if ((DEBUG & (DEBUG_FIELD_REFERENCE | DEBUG_SOURCE)) != 0)
			System.out.println
				("    PVL_to_DB.Database_Reference: Retry with source parameters" + NL
				+"    after exception -" + NL
				+ unresolved_reference.getMessage ());

		/*	Retry

			Bind the Reference_Resolver to the retry_parameters
			supplemented by the currently bound parameters.
		*/
		Parameter
			config_parameters = The_Resolver.Parameters (),
			parameters = null,
			source_params = null;
		try
			{
			parameters = new Parameter (config_parameters);
			source_params = new Parameter (retry_parameters);
			if (source_params.Is_Aggregate ())
				parameters.Add (source_params.List ());
			else
				parameters.Add (source_params);
			}
		catch (PVL_Exception e) {}
		The_Resolver.Parameters (parameters);

		//	Ensure no default value for source parameter reference resolution.
		String
			default_value = The_Resolver.Default_Value ();
		The_Resolver.Default_Value (null);

		try {database_reference = The_Resolver.Resolve (map_parameter.Name ());}
		catch (Exception except)
			{
			if ((DEBUG & (DEBUG_FIELD_REFERENCE | DEBUG_SOURCE)) != 0)
				System.out.println
					("    Retry failed -" + NL
					+ except.getMessage ());
			if (! except.getMessage ().equals (message))
				unresolved_reference = new Unresolved_Reference (Error_Message
					("For the database reference of map entry" + NL
					+ (map_parameter.Is_Aggregate () ?
						(map_parameter.Name () + NL) :
						map_parameter.Description ())
					+ message + NL
					+ "After retry with source parameters -" + NL
					+ except.getMessage ()),
					map_parameter.Name ());
			throw unresolved_reference;
			}
		finally
			{
			//	Restore The_Resolver nested reference context.
			The_Resolver.Default_Value (default_value);
			The_Resolver.Parameters (config_parameters);
			}
		if ((DEBUG & (DEBUG_FIELD_REFERENCE | DEBUG_SOURCE)) != 0)
			System.out.println ("    Retry succeeded");
		}
	else
		throw unresolved_reference;
	}
if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
	System.out.println
		("<<< PVL_to_DB.Database_Reference: " + database_reference);
return database_reference;
}

/**	Gets a database reference from a map parameter.
<p>
	@param	map_parameter	The map parameter which has a database
		reference as its name.
	@return	The resolved database reference String.
	@throws	ParseException	if a nested reference could not be resolved due
		to invalid syntax.
	@throws	Database_Exception	if there was a problem accessing
		the database to resolve a nested database reference.
	@throws	Unresolved_Reference	if a nested reference could not be
		resolved and there was no {@link #Default_Value(String)
		Default_Value}.
	@see	#Database_Reference(Parameter, Parameter)
*/
private String Database_Reference
	(
	Parameter	map_parameter
	)
	throws
		ParseException,
		Database_Exception,
		Unresolved_Reference
{return Database_Reference (map_parameter, null);}

/**	Gets a table reference from a database reference.
<p>
	@param	database_reference	The database reference String from which
		to extract the table reference.
	@return	The catalog.table reference, or null if no table reference
		is present (maybe just a field name).
	@see	Reference_Resolver#Table_Name(String)
	@see	Database#Table_Reference(String)
	@see	Reference_Resolver#Catalog_Name(String)
*/
private String Table_Reference
	(
	String		database_reference
	)
	throws Database_Exception
{
if ((DEBUG & (DEBUG_FIELD_REFERENCE | DEBUG_SOURCE)) != 0)
	System.out.println
		(">>> PVL_to_DB.Table_Reference: " + database_reference);
/*	Add the table name to the catalog name from the database reference.
	N.B.: If there is no table name, there will be no catalog name.
*/
String
	table_reference = The_Resolver.Table_Name (database_reference);
if (table_reference != null)
	{
	try {table_reference = The_Database.Table_Reference
			(The_Resolver.Catalog_Name (database_reference), table_reference);}
	catch (Database_Exception exception)
		{
		/* Shouldn't happen. */
		table_reference = null;
		}
	}
if ((DEBUG & (DEBUG_FIELD_REFERENCE | DEBUG_SOURCE)) != 0)
	System.out.println
		("<<< PVL_to_DB.Table_Reference: " + table_reference);
return table_reference;
}

/**	Resolve all references in a Parameter Value.
<p>
	The Parameter must be an Assignment with a Value that is not an
	Array. The Value as a String is submitted to {@link
	#Resolve_Source_Reference(String)}, with handlers for all exception
	conditions.
<p>
	@param	parameter	The map Parameter to have its Value resolved.
	@return	The resolved String, or null if there was no resolution.
	@throws	Invalid_Map_Syntax	If the parameter is not an Assignment,
		its Value is an Array, or a unresolved source reference resulted
		in a forced exception.
	@throws	Database_Exception	From {@link Resolve_Source_References(String)}.
	@throws	ParseException	From {@link Resolve_Source_References(String)}.
	@throws	Unresolved_Reference From {@link Resolve_Source_References(String)}.
*/
private String Resolve_Source_Reference
	(
	Parameter	parameter
	)
	throws
		Invalid_Map_Syntax,
		ParseException,
		Database_Exception,
		Unresolved_Reference
{
if (! parameter.Is_Assignment ())
	throw new Invalid_Map_Syntax (Error_Message
		("Invalid " + parameter.Classification_Name () + " map parameter."),
		null, parameter);
try 
	{
	if (parameter.Value ().Is_Array ())
		throw new Invalid_Map_Syntax (Error_Message
			("Invalid " + parameter.Value ().Type_Name ()
				+ " of values for a map parameter."),
			null, parameter);
	}
catch (PVL_Exception exception) {/* Checked for Aggregate above. */}

try
	{return Resolve_Source_References (parameter.Value ().String_Data ());}
catch (PVL_Exception exception) {/* Checked for Array above. */}
catch (Database_Exception exception)
	{
	throw new Database_Exception
		(exception.getMessage () + NL
		+"Map parameter: " + parameter.Description ());
	}
catch (ParseException exception)
	{
	throw new ParseException
		(exception.getMessage () + NL
		+"Map parameter: " + parameter.Description (),
		exception.getErrorOffset ());
	}
catch (Unresolved_Reference exception)
	{
	throw new Unresolved_Reference
		(exception.getMessage () + NL
		+"Map parameter: " + parameter.Description ());
	}
catch (Invalid_Map_Syntax exception)
	{
	exception.Parameter (parameter);
	throw exception;
	}
return null;
}

/**	Resolves a Source reference to its value.
<p>
	A PVL source reference, as occurs in {@link #Map(String) Map}
	Parameter Values, is resolved to a value. The steps taken to achieve a
	resolved value are:
<p>
	The source reference is first resolved of all nested references. If
	there are no syntax errors or an exception due to an unresolved
	reference the result is evaluated for individual reference tokens.
<p>
	The source reference expression is searched for individual reference
	tokens. Each identified individual reference is resolved to a string
	value. The resolved values are concatenated in the left-to-right
	order they occur in the source reference to produce its resolved
	value. <b>N.B.</b>: Unquoted whitespace characters (any of " \t\r\n")
	are compressed out of the expression during the process of
	resolution value accumulation.
<p>
	Individual references are identified and resolved as follows, in the
	order in which they are listed:
<dl>
<dt>A logically empty string.
	<dd>The source reference is trimmed of leading and trailing
		whitespace characters. If the result is an empty string the
		resolved value is an empty string.
<dt>A mathematical expression.
	<dd>An attempt is made to evaluate the source reference as a
		mathematical expression. If this succeeds the result, as a
		string representation, is the resolved value.
		<p>
		The expression {@link edu.hws.jcm.data.Parser#Parser() parser}
		accepts both simple and complex syntax with numeric constants
		plus the symbolic constants pi and e, the usual mathematical
		operators (unary and binary + and -, * and /) plus exponentiation
		(either ^ or **) and a full complement of standard math functions
		(sin, cos, tan, cot, sec, csc, arcsin, arccos, arctan, exp, ln,
		log2, log10, sqrt, cubert, abs, round, floor, ceiling and trunc).
		Also, logical operators (&, |, ~, =, <, >, <>, <= and >=; the
		words "and", "or", and "not" can be used in place of &, | and ~)
		may be used in a conditional expression (with ? and :) as long as
		the result of the expression evaluation is a single numeric
		value. The evaluation will use double precision arithmetic and
		the result will be provided in its {@link Double#toString(double)
		string representation} form.
		<p>
		<b>N.B.</b>: If the results of resolving a mathematical expression
		are equivalent to an integer, despite being returned as a
		double type, then the resolved value will be represented as an
		integer without a ".0" suffix.
		<p>
		<b>N.B.</b>: Mathematical expression evaluation of an individual
		reference can be suppressed by preceding the reference with the
		{@link #NO_MATH_MARKER} character. In this case the marker is
		stripped off an no attempt is made to evaluate the reference or
		its resolved value as a mathematical expression.
<dt>A quoted string.
	<dd>All characters within unescaped single or double quotes are taken
		verbatim. The quotation marks are stripped off and the resulting
		string is the resolved value. <b>N.B.</b>: Quoted strings do not
		need to be surrounded by whitespace; they may be embedded anywhere
		in a source reference. Conversely, to include whitespace characters
		in the resolved value of a source reference they must be quoted.
		<p>
		A quotation mark without a closing unescaped matching quotation
		mark in the same source reference results in an {@link
		Invalid_Map_Syntax} exception being thrown that identifies the
		unclosed quoted string as the unresolved reference. 
<dt>A parenthesized expression.
	<dd>The characters within unescaped parentheses are recursively
		processed as a source reference to produce a resolved value. The
		enclosing parentheses are stripped off. Nested parentheses are
		included and, unless escaped, will be recognized during the
		recursive procssing of the new source reference. <b>N.B.</b>:
		Parenthetical expressions do not need to be surrounded by
		whitespace; they may be embedded anywhere in a source reference.
		<p>
		An opening parenthesis without a closing unescaped matching
		closing parenthesis, ignoring nested parentheses, in the same
		source reference results in an {@link Invalid_Map_Syntax}
		exception being thrown that identifies the unclosed parenthetical
		string as the unresolved reference. 
<dt>An alternative marker.
	<dd>The occurance of an {@link #ALTERNATIVE_MARKER} character causes
		all further accumulation of resolved references in the current
		source reference to stop if, and only if, the immediately
		preceeding source reference (if any) was successfully resolved.
		That is, if the immediately preceeding reference fails to resolve
		then the occurance of an alternative marker causes the failed
		reference to be ignored.
		<p>
		<b>N.B.</b>: An alternative marker may be followed by any
		source reference other than an alternative marker, including
		an empty string. If another alternative marker follows in the
		current source reference then an {@link Invalid_Map_Syntax}
		exception is thrown that identifies the current source reference.
<dt>A source parameter reference.
	<dd>A contiguous sequence of non-whitespace characters that are not
		quoted or parenthesized and is not an alternative marker is
		evaluated as a source parameter reference. A source parameter
		reference is resolved against the current {@link
		#Source_Parameters(Parameter) source parameters}, with the
		{@link #Default_Parameters(Parameter) default parameters}
		appended, to produce a resolved value.
		<p>
		If the source parameter reference is unresolved <i>and</i> the
		next reference in the current source reference is an
		alternative marker the unresolved reference is ignored and
		parsing continues with the next token. Note that if an
		alternative marker occurs at the end of a source reference it
		will never fail to resolve to at least the empty string value.
		However, if an
		alternative marker does not follow a source parameter reference
		that fails to resolve to a value then further parsing of the
		source reference stops and the enclosing source reference, if
		any, is signaled that an unresolved source reference occured;
		this source reference may be followed by an alternative marker.
		This enables any parenthesized expression to be provided with
		an alternative in case the entire parenthesized expression fails
		to resolve.
</dl><p>
	After each individual source reference has been successfully
	resolved another attempt is made to evaluate the results as a
	mathematical expression, unless this has been suppressed by the use
	of the {@link #NO_MATH_MARKER} character in front of the reference.
	If this succeeds, the results are used as the resolved value;
	otherwise the original resolved value is used. <b>N.B.</b>: To
	include parameter source references in post-resolution mathematical
	expressions all parts of the expression that are <i>not</i> to be
	otherwise treated as resolvable references must be quoted. This is
	an admittedly awkward requirement, but otherwise all parameter
	source references would need special syntax identification which
	would be even more awkward.
<p>
	@param	source_reference	The reference String to be resolved to a
		Parameter source value.
	@return	The resolved String value.
	@throws	ParseException	If the evaluation of a mathematical expression
		in a nested reference encounters a syntax error.
	@throws Database_Exception	If a nested database reference can not be
		resolved.
	@throws	Unresolved_Reference	If a nested reference can not be
		resolved and there is no {@link #Default_Value(String) default
		reference value}.
	@throws	Invalid_Map_Syntax	If a parameter source reference can not
		be resolved, or a source parameter reference alternative is the
		quoted string constant {@link
		Conductor#UNRESOLVED_REFERENCE_THROWS "THROW"} (case
		insensitive).
*/
public String Resolve_Source_References
	(
	String	source_reference
	)
	throws
		ParseException,
		Database_Exception,
		Unresolved_Reference,
		Invalid_Map_Syntax
{
if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
	System.out.println
		(">>> PVL_to_DB.Resolve_Source_References:" + NL
		+"    source reference: " + source_reference);

//	Resolve any embedded references in the source reference.
String
	reference = null;
try {reference = The_Resolver.Resolve (source_reference);}
catch (ParseException exception)
	{
	throw new ParseException (Error_Message
		(exception.getMessage () + NL
		+"In source parameter reference \"" + source_reference + '"' + NL
		+"at location " + exception.getErrorOffset ()),
		exception.getErrorOffset ());
	}
catch (Database_Exception exception)
	{
	throw new Database_Exception (Error_Message
		(exception.getMessage () + NL
		+"In source parameter reference \"" + source_reference + '"'));
	}
catch (Unresolved_Reference exception)
	{
	throw new Unresolved_Reference (Error_Message
		(exception.getMessage () + NL
		+"In source parameter reference \"" + source_reference + '"'));
	}
if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
	System.out.println
		("    PVL_to_DB.Resolve_Source_References:"
		+" Source pattern resolved to - " + reference);

//	Bind the Reference_Resolver to the Source_Parameters.
The_Resolver.Parameters (Source_Parameters);

//	Ensure no default value for source parameter reference resolution.
String
	default_value = The_Resolver.Default_Value ();
The_Resolver.Default_Value (null);

String
	value = null;
try {value = Resolve_Source_Reference (reference);}
catch (Invalid_Map_Syntax exception)
	{
	if (! reference.equals (exception.Reference ()))
		throw new Invalid_Map_Syntax
			((exception.getMessage () + NL
			+"in source parameter reference \"" + source_reference + "\""),
			exception.Reference ());
	throw exception;
	}
catch (Unresolved_Reference exception)
	{
	if (No_Op)
		System.out.println
			("    Unresolved reference: " + exception.Reference ());
	else if (Unresolved_Source_Reference_Throws)
		{
		if (! reference.equals (exception.Reference ()))
			throw new Unresolved_Reference
				((exception.getMessage () + NL
				+"in source parameter reference \"" + source_reference + "\""),
				exception.Reference ());
		throw exception;
		}
	}
catch (ParseException exception)
	{
	throw new ParseException
		((exception.getMessage () + NL
		+"in source parameer reference \"" + source_reference + "\""),
		exception.getErrorOffset ());
	}
finally
	{
	//	Restore The_Resolver nested reference context.
	The_Resolver.Default_Value (default_value);
	The_Resolver.Parameters (The_Database.Configuration ());
	}

if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
	System.out.println
		("<<< PVL_to_DB.Resolve_Source_References: " + value);
return value;
}

/**	Identify and resolve each individual reference in a source reference.
<p>
	The source reference is searched for individual references. Each
	identified individual reference is resolved to a String value. The
	resolved values are appended contiguously to produce the resolved
	value of the source reference.
<p>
	@param	source_reference	A source reference String to be resolved.
	@return	The resolved value String.
	@throws	Unresolved_Reference	If the current source reference
		could not be resolved.
	@throws	Invalid_Map_Syntax	If an unclosed quoted or parenthetical
		string occurs, or an alternative marker is followed by another
		alternative marker.
	@throws	ParseException	If there was a mathematical expression parsing
		problem during reference resolving.
	@throws	Unresolved_Reference	If the reference could not be resolved.
	@see	#Resolve_Source_References(String)
*/
private String Resolve_Source_Reference
	(
	String	source_reference
	)
	throws
		ParseException,
		Unresolved_Reference,
		Invalid_Map_Syntax
{
if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
	System.out.println
		(">>> PVL_to_DB.Resolve_Source_Reference: "
			+ source_reference);
String
	reference = source_reference.trim ();
if (reference.length () == 0)
	{
	System.out.println
		("<<< PVL_to_DB.Resolve_Source_Reference: Empty");
	return reference;
	}

boolean
	evaluate_for_math_expression = true;
if (reference.charAt (0) == NO_MATH_MARKER)
	{
	evaluate_for_math_expression = false;
	reference = reference.substring (1);
	if (reference.length () == 0)
		return reference;
	}

if (evaluate_for_math_expression)
	{
	//	Try to evaluate it as a mathematical expression.
	try
		{
		reference = number_representation
			(The_Resolver.Evaluate_to_double (reference));
		if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
			System.out.println
				("    Math expression evaluated" + NL
				+"<<< PVL_to_DB.Resolve_Source_Reference: " + reference);
		return reference;
		}
	catch (ParseException parse_exception) {}
	}

String
	resolved = "",
	value = "";
Unresolved_Reference
	unresolved_reference = null;
Words
	words = new Words (reference)
		.Quoted_Words (true)
		.Parenthesized_Words (true);

Scan_for_References:
while ((reference = words.Next_Word ()).length () != 0)
	{
	if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
		System.out.println
			("    > Reference - " + reference);
	switch (reference.charAt (0))
		{
		case '"':
		case '\'':
			//	Quoted string.
			if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
				System.out.println
					("    Quoted string - " + reference);
			unresolved_reference = null;	//	Got an alternative.
			if (reference.length () == 1 ||
				reference.charAt (reference.length () - 1)
					!= reference.charAt (0))
				throw new Invalid_Map_Syntax (Error_Message
					("Quoted string is not closed."),
					reference);
			value = reference.substring (1, reference.length () - 1);
			break;

		case '(':
			//	Parenthesized expression.
			if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
				System.out.println
					("    Parenthesized expression - " + reference);
			unresolved_reference = null;	//	Got an alternative.
			if (reference.length () == 1 ||
				reference.charAt (reference.length () - 1) != ')')
				throw new Invalid_Map_Syntax (Error_Message
					("Parenthesized expression is not closed."),
					reference);
			try {value = Resolve_Source_Reference
					(reference.substring (1, reference.length () - 1));}
			catch (Unresolved_Reference exception)
				{unresolved_reference = exception;}
			break;

		case ALTERNATIVE_MARKER:
			if (unresolved_reference != null)
				//	Already looking for an alternative.
				throw new Invalid_Map_Syntax (Error_Message
					("Sequential alternative source reference markers ("
						+ ALTERNATIVE_MARKER + ')' + NL
					+"after unresolved reference -" + NL
					+ unresolved_reference.getMessage ()),
					source_reference);

			//	Not looking for an alternative. Done.
			break Scan_for_References;

		default:
			//	Parameter source reference.
			if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
				System.out.println
					("    Source reference - " + reference);
			unresolved_reference = null;	//	Got an alternative.
			try
				{
				value =
					The_Resolver.Resolve_Parameter_Reference (reference);
				if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
					System.out.println
						("    Resolved reference - " + value);
				break;
				}
			catch (ParseException exception)
				{
				throw new ParseException (Error_Message
					(exception.getMessage () + NL
					+"At location " + exception.getErrorOffset () + NL
					+"in source reference \"" + reference + "'"
					+ (source_reference.equals (reference) ?
						"" : (NL + "of \"" + source_reference + '"'))),
					exception.getErrorOffset ());
				}
			catch (Unresolved_Reference exception)
				{unresolved_reference = exception;}
			break;
		}
	if (unresolved_reference != null)
		{
		if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
			System.out.println
				("    Reference did not resolve");
		//	Check for an alternative reference marker.
		reference = words.Next_Word ();
		if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
			System.out.println
				("    Next word - " + reference);
		if (reference.length () == 0 ||
			reference.charAt (0) != ALTERNATIVE_MARKER ||
			reference.length () > 2 ||
			(reference.length () == 2 &&
			 reference.charAt (1) != ALTERNATIVE_MARKER))
			{
			//	No alternative.
			if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
				System.out.println
					("<<< PVL_to_DB.Resolve_Source_Reference: Unresolved");
			throw unresolved_reference;
			}

		//	Take the alternative.
		if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
			System.out.println
				("    Alternative ...");
		continue;
		}

	//	Accumulate the resolved reference.
	if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
		System.out.println
			("    Resolved reference - " + value);
	resolved += value;
	}
if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
	System.out.println
		("    Resolution: " + resolved);


if (evaluate_for_math_expression)
	{
	//	Try to evaluate it as a mathematical expression (just in case :-).
	try
		{
		resolved = number_representation
			(The_Resolver.Evaluate_to_double (resolved));
		if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
			System.out.println
				("    Math expression evaluated");
		}
	catch (ParseException e) {}
	}

if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_SOURCE)) != 0)
	System.out.println
		("<<< PVL_to_DB.Resolve_Source_Reference: " + resolved);
return resolved;
}

/**	Get the number representation of a double value.

	If the value is equivalent to an integer it's number representation
	is as the integer; i.e. no ".0" suffix is included.

	@param	value	The double to be represented.
	@return	The String representation of the value.
*/
private static String number_representation
	(
	double	value
	)
{
if (value == (int)value)
	return String.valueOf ((int)value);
return String.valueOf (value);
}

/*==============================================================================
	Helpers
*/
/**	Assembles an Aggregate by parsing a PVL source.
<p>
	@param	source	A PVL source String. This may be a URL, local
		pathname, or a system resource.
	@return	An Aggregate Parameter. The Aggregate will be given the
		source Name. If the source is null, null will be returned.
	@throws	PVL_Exception	If there was a problem reading from or
		parsing the source stream.
	@see	Streams#Get_Stream(String)
*/
private static Parameter Assemble_PVL
	(
	String	source
	)
	throws PVL_Exception
{
if (source == null)
	return null;
if ((DEBUG & DEBUG_MAP) != 0)
	System.out.println
		(">>> PVL_to_DB.Assemble_PVL: " + source);
InputStream
	source_stream = Streams.Get_Stream (source);
if (source_stream == null)
	throw new PVL_Exception (Error_Message (
		"Unable to access source " + source + NL
		+ "  to assemble the parameters map."));
Parameter
	aggregate;
try
	{
	aggregate = new Parameter (new Parser (source_stream)
		.Strict (false)
		.Crosshatch_Comments (true));
	}
catch (PVL_Exception exception)
	{
	throw new PVL_Exception (Error_Message (
		"Unable to read the PVL to database map from the \""
			+ source + "\" source." + NL
		+ exception.getMessage ()));
	}
aggregate.Name (source);
if ((DEBUG & DEBUG_MAP) != 0)
	System.out.println
		("<<< PVL_to_DB.Assemble_PVL:" + NL
		+ aggregate.Description ());
return aggregate;
}

/**	Produces an error message with the application's <code>ID</code>.
<p>
	The message string will be prepended with the class {@link #ID ID}
	and a new-line, if the ID is not already present in the message. If
	the message is null, it will be replaced with the ID.
<p>
	@param	message	The message String.
	@return	The modified message String.
*/
private static String Error_Message
	(
	String			message
	)
{
if (message == null)
	message = ID;
else if (message.indexOf (ID) < 0)
	message =
		ID + NL
		+ message;
return message;
}

/**	Produces a Database_Exception containing a message.
<p>
	The message will be prepended with a description of the Database
	from the Configuration and then modified by the {@link
	#Error_Message(String) Error_Message} method before it is used to
	construct a new Database_Exception.
<p>
	@param	message	The message String.
	@return	A Database_Exception.
*/
private Database_Exception Database_Error
	(
	String	message
	)
{
if (The_Database == null)
	message =
		"No Database specified." + NL
		+ message;
else
	{
	Configuration
		configuration = The_Database.Configuration ();
	message =
		"Problem accessing " + configuration.Get_One (Database.TYPE)
			+ " database on host " + configuration.Get_One (Configuration.HOST)
			+ ":" + NL
		+ message;
	}
return new Database_Exception (Error_Message (message));
}

/*==============================================================================
	Application main
*/
/**	Runs a PVL_to_DB application.
<p>
	<b>N.B.</b>: If a Database connection is establshed it is always
	Disconnected before the application exits for any reason.
<p>
	Exit status values:
<p>
	<dl>
	<dt>{@link #EXIT_SUCCESS}
		<dd>Success
	<dt>{@link #EXIT_INVALID_COMMAND_LINE_SYNTAX}
		<dd>Invalid command line syntax
	<dt>{@link #EXIT_CONFIGURATION_PROBLEM}
		<dd>Configuration problem
	<dt>{@link #EXIT_DATABASE_ERROR}
		<dd>Database access error
	<dt>{@link #EXIT_PVL_ERROR}
		<dd>PVL processing problem
	<dt>{@link #EXIT_INVALILD_SYNTAX}
		<dd>Invalid syntax was found in the map
	<dt>{@link #EXIT_UNRESOLVED_REFERENCE}
		<dd>An unresolved reference was encountered
	</dl>
<p>
	@param	arguments	The {@link #Usage command line} arguments.
*/
public static void main
	(
	String []	arguments
	)
{
if (arguments.length == 0)
	Usage (true);
PVL_to_DB
	mapper = null;
String
	option,
	configuration_source = null,
	database_server = null,
	map_source = null,
	name;
Vector
	PVL_sources = new Vector ();
boolean
	verbose = false,
	noop = false,
	unresolved_source_reference_throws = false;
if (DEBUG != 0)
	verbose = true;
int
	looping = 0,
	matching = Reference_Resolver.CASE_SENSITIVE;
Configuration
	user_parameters = null;
try {user_parameters = new Configuration ((Parameter)null);}
catch (Configuration_Exception exception) {}

for (int count = 0;
	 	 count < arguments.length;
	 	 count++)
	{
	if (arguments[count].length () == 0)
		continue;
	if (arguments[count].charAt (0) == '-' &&
		arguments[count].length() > 1)
		{
		option = arguments[count];
		if (option.equalsIgnoreCase ("-SERVER"))
			option = "-Database";
		switch (option.charAt (1))
			{
			case 'C':	//	-Configuration
			case 'c':
				if (++count == arguments.length ||
					arguments[count].length () == 0 ||
					arguments[count].charAt (0) == '-')
					{
					//	Use the default configuration file.
					name = DEFAULT_CONFIGURATION_FILENAME;
					--count;
					}
				else
					name = arguments[count];
				if (configuration_source == null)
					configuration_source = name;
				else
					{
					System.out.println
						("Multiple configuration sources:" + NL
						+configuration_source + NL
						+"and" + NL
						+name);
					Usage (verbose);
					}
				break;
			case 'D':	//	-Database
			case 'd':
				if (++count == arguments.length ||
					arguments[count].length () == 0 ||
					arguments[count].charAt (0) == '-')
					{
					System.out.println
						("Missing database server name ");
					Usage (verbose);
					}
				name = arguments[count];
				if (database_server == null)
					database_server = name;
				else
					{
					System.out.println
						("Multiple database servers: "
							+ database_server + " and " + name);
					Usage (verbose);
					}
				break;
			case 'H':	//	-Help
			case 'h':
				Usage (true);
			case 'L':	//	-Loop
			case 'l':
				if (++count == arguments.length ||
					arguments[count].length () == 0 ||
					arguments[count].charAt (0) == '-')
					looping = 0;
				else
					{
					try {looping = Integer.parseInt (arguments[count]);}
					catch (NumberFormatException exception)
						{
						System.out.println
							("Loop count expected; but \""
								+ arguments[count] + "\" was found.");
						Usage (verbose);
						}
					}
				break;
			case 'M':	//	-Map
			case 'm':
				if (++count == arguments.length ||
					arguments[count].length () == 0 ||
					arguments[count].charAt (0) == '-')
					{
					System.out.println
						("Missing map source.");
					Usage (verbose);
					}
				name = arguments[count];
				if (map_source == null)
					map_source = name;
				else
					{
					System.out.println
						("Multiple map sources:" + NL
						+ map_source + NL
						+"and" + NL
						+ name);
					Usage (verbose);
					}
				break;
			case 'N':	//	-Noop
			case 'n':
				noop = true;
				break;
			case 'P':	//	-Pattern
			case 'p':
				if (++count == arguments.length ||
					arguments[count].length () == 0 ||
					arguments[count].charAt (0) == '-')
					{
					System.out.println
						("Missing pattern match selection.");
					Usage (verbose);
					}
				switch (arguments[count].charAt (0))
					{
					case 'C':	//	Case sensitive.
					case 'c':
						matching = Reference_Resolver.CASE_SENSITIVE;
						break;
					case 'I':	//	Ignore case.
					case 'i':
						matching = Reference_Resolver.CASE_INSENSITIVE;
						break;
					case 'R':	//	Regex.
					case 'r':
						matching = Reference_Resolver.PATTERN_MATCH;
						break;
					default:
						System.out.println
							("Unrecognized pattern match selection: "
								+ arguments[count]);
						Usage (verbose);
					}
				break;
			case 'Q':	//	-Quiet
			case 'q':
				if (DEBUG == 0)
				verbose = false;
				break;
			case 'S':	//	-Set
			case 's':
				if (++count == arguments.length ||
					arguments[count].length () == 0 ||
					arguments[count].charAt (0) == '-')
					{
					System.out.println
						("Missing parameter definition to set.");
					Usage (verbose);
					}
				try {user_parameters.Set
						(new Parameter (new Parser (arguments[count])));}
				catch (Exception exception)
					{
					System.out.println
						("Unable to set the parameter definition: \""
							+ arguments[count] + '"' + NL
						+ exception.getMessage ());
					System.exit (EXIT_INVALID_COMMAND_LINE_SYNTAX);
					}
				break;
			case 'u':
			case 'U':
				unresolved_source_reference_throws = true;
				break;
			case 'V':	//	-Verbose
			case 'v':
				verbose = true;
				break;
			default:
				System.out.println
					("Unrecognized command line argument: "
						+ arguments[count]);
				Usage (true);
			}
		}
	else
		PVL_sources.add (arguments[count]);
	}

//	Check requirements.
if (map_source == null)
	{
	System.out.println
		("No map source.");
	Usage (verbose);
	}
if (PVL_sources.size () == 0)
	{
	System.out.println
		("No PVL source file(s).");
	Usage (verbose);
	}

//	Construct the Configuration.
Configuration
	configuration = null;
if (configuration_source == null)
	configuration_source = DEFAULT_CONFIGURATION_FILENAME;
try {configuration = new Configuration
	(configuration_source);}
catch (Configuration_Exception exception)
	{
	System.out.println (ID + NL
		+ "Unable to construct a Configuration from the" + NL
		+ configuration_source + " source." + NL
		+ exception.getMessage ());
	System.exit (EXIT_CONFIGURATION_PROBLEM);
	}

//	Set user parameters from the command line.
try {configuration.Set (user_parameters);}
catch (Configuration_Exception exception)
	{
	System.out.println (ID + NL
		+ "Unable to apply the parameter(s) set on the command line." + NL
		+ exception.getMessage ());
	System.exit (EXIT_CONFIGURATION_PROBLEM);
	}

//	Establish the Database connection.
Database
	database = null;
try
	{
	database = new Database (configuration);
	database.Connect (database_server);
	}
catch (Database_Exception exception)
	{
	System.out.println (ID + NL
		+ "Unable to connect to the database." + NL
		+ exception.getMessage ());
	System.exit (EXIT_DATABASE_ERROR);
	}
catch (Configuration_Exception exception)
	{
	System.out.println (ID + NL
		+ "Unable to connect to the database." + NL
		+ exception.getMessage ());
	System.exit (EXIT_CONFIGURATION_PROBLEM);
	}

//	Construct the PVL_to_DB object.
try {mapper = new PVL_to_DB (database, map_source);}
catch (Configuration_Exception exception)
	{
	System.out.println (ID + NL
		+ "Unable to configure the PVL_to_DB mapper." + NL
		+ exception.getMessage ());
	Disconnect (mapper);
	System.exit (EXIT_CONFIGURATION_PROBLEM);
	}
catch (PVL_Exception exception)
	{
	System.out.println (ID + NL
		+ "Unable to initialize the PVL_to_DB mapper." + NL
		+ exception.getMessage ());
	Disconnect (mapper);
	System.exit (EXIT_PVL_ERROR);
	}

//	Set the modes.
mapper
	.No_Op (noop)
	.Match_Mode (matching)
	.Unresolved_Source_Reference_Throws (unresolved_source_reference_throws)
	.Verbose (verbose);
if (looping > 1)
	{
	mapper.Loop_Limit = looping;
	mapper.Loop (true);
	}

//	Map the PVL files into the database.
if (verbose)
	System.out.println (ID);
String
	source = null;
int
	records = 0;
try
	{
	Iterator
		sources = PVL_sources.iterator ();
	while (sources.hasNext ())
		records += mapper.PVL_Source (source = (String)sources.next ());
	}
catch (Configuration_Exception exception)
	{
	System.out.println (ID + NL
		+ "Configuration problem while mapping source:" + NL
		+ source + NL
		+ exception.getMessage ());
	Disconnect (mapper);
	System.exit (EXIT_CONFIGURATION_PROBLEM);
	}
catch (PVL_Exception exception)
	{
	System.out.println (ID + NL
		+ "PVL problem while mapping source:" + NL
		+ source + NL
		+ exception.getMessage ());
	Disconnect (mapper);
	System.exit (EXIT_PVL_ERROR);
	}
catch (ParseException exception)
	{
	System.out.println (ID + NL
		+ "Math expression problem in reference while mapping source:" + NL
		+ source + NL
		+ exception.getMessage ());
	Disconnect (mapper);
	System.exit (EXIT_INVALILD_SYNTAX);
	}
catch (Invalid_Map_Syntax exception)
	{
	System.out.println (ID + NL
		+ "Invalid map syntax encountered while mapping source:" + NL
		+ source + NL
		+ exception.getMessage ());
	String
		message;;
	if ((message = exception.Pathname ()) != null)
		System.out.println ("Map parameter name: " + message);
	if ((message = exception.Value ()) != null)
		System.out.println ("Map parameter value: " + message);
	if ((message = exception.Reference ()) != null)
		System.out.println ("Invalid reference: " + message);
	if (exception.getCause () != null)
		System.out.println ("Exception cause: "
			+ exception.getCause ().getMessage ());
	Disconnect (mapper);
	System.exit (EXIT_INVALILD_SYNTAX);
	}
catch (Database_Exception exception)
	{
	System.out.println (ID + NL
		+ "Database problem while mapping source:" + NL
		+ source + NL
		+ exception.getMessage ());
	Disconnect (mapper);
	System.exit (EXIT_DATABASE_ERROR);
	}
catch (Unresolved_Reference exception)
	{
	System.out.println (ID + NL
		+ "Unresolved reference encountered while mapping source:" + NL
		+ source + NL
		+ exception.getMessage ());
	Disconnect (mapper);
	System.exit (EXIT_UNRESOLVED_REFERENCE);
	}

if (verbose)
	System.out.println
		(NL
		+"Grand total records affected: " + records);
Disconnect (mapper);
System.exit (EXIT_SUCCESS);
}

/**	Disconnects the Database of a PVL_to_DB.
<p>
	If an exception is thrown by the Database Disconnect the exception
	message is reported to stderr.
<p>
	@param	pvl_to_DB	A PVL_to_DB object. This may be null in which case
		nothing is done.
*/
private static void Disconnect
	(
	PVL_to_DB pvl_to_DB
	)
{
if (pvl_to_DB == null)
	return;
try
	{
	pvl_to_DB.Database ().Disconnect ();
	}
catch (Database_Exception exception)
	{
	System.err.println (ID + NL
		+ "Unable to disconnect from the database." + NL
		+ exception.getMessage ());
	}
}

/**	Prints the command-line usage syntax.
<p>
<blockquote><pre>
Usage: <b>PVL_to_DB</b> &lt;<i>Options</i>&gt; <b>-<u>M</u>ap</b> &lt;<i>source</i>&gt; &lt;<i>PVL source</i>&gt; [...]
&nbsp;&nbsp;Options -
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>D</u>atabase</b>|<b>-<u>SERVER</u></b> &lt;<i>server name</i>&gt;
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>C</u>onfiguration</b> [&lt;<i>source</i>&gt;] (default: Database.conf)
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>S</u>et</b> &lt;<i>parameter definition</i>&gt;
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>L</u>oop</b> [&lt;<i>count</i>&gt;] (default: 0, no looping)
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>P</u>attern</b> <b><u>C</u>ase_sensitive</b> | <b><u>I</u>gnore_case</b> | <b><u>R</u>egex</b> (default: Case_sensitive)
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>V</u>erbose</b> (default: false)
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>Q</u>uiet</b> (default: true)
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>N</u>o-op</b> (default: false)
</pre></blockquote>
<p>
<h4>
	PVL source:
</h4><p>
	The pathname or URL of a file containing PVL to be used as the source
	of parameter values to be mapped into database field values.
<p>
<h4>
	Map:
</h4><p>
	The pathname or URL of a file containing a {@link #Map(String) map}
	of PVL parameters to database fields.
<p>
<h4>
	Database server name:
</h4><p>
	The Configuration file may contain connection information for more
	than one database. The information for each database is organized
	by {@link Database#SERVER server} name, which may be specified. If
	no server name is provided, the Database will attempt to use the
	default server specified in the Configuration.
<p>
<h4>
	Configuration:
</h4><p>
	If the pathname or URL of the {@link PIRL.Configuration.Configuration
	Configuration} file is not specified the filename "Database.conf"
	will be used by default. If the configuration file is not in the
	current working directory, it will be looked for in the user's home
	directory. The configuration file must contain the necessary
	information needed to identify and connect with the database server
	(as required by the {@link
	PIRL.Database.Database#Database(Configuration) Database} constructor
	and its {@link PIRL.Database.Database#Connect() Connect} method).
	These are typically the database server "Type", "Host", "User" and
	"Password" access values.
<p>
	The configuration parameters are used to resolve "nested" parameter
	references (see the discussion of {@link #Map(String) map} parameters
	for the distinction between nested configuration parameter referenes
	and source parameter references). In addition, if the configuration
	file contains a {@link #PVL_to_DB_GROUP PVL_to_DB} Parameter group,
	then these parameters will be included in the {@link
	#Default_Parameters(Parameter) defaults} list to be used when
	resolving source parameter references.
<p>
<h4>
	Set:
</h4><p>
	Parameters defined on the command line supplement or override
	existing  configuration file parameters. Parameters are specified
	using the usual {@link Parameter PVL} syntax. <b>N.B.</b>: Spaces,
	quotes and other special characters that are part of the parameter
	definition must be protected from interpretation by the command
	shell. Parameters may have simple - typically single word - names or
	absolute pathnames including groups that exist in the configuration
	file or new groups to be created. This option may be used repeatedly
	to define multiple paramters.
<h4>
	Loop:
</h4><p>
	Normally only one mapping pass is made through the PVL source
	parameters. If a {@link #Loop(boolean) loop} count greater than 1
	is specified then each PVL source parameters set will be mapped
	repeatedly up to the loop count or until no database updates occur.
	This allows sets of parameters having the same names to be
	collected and sequentially inserted into the database using the
	{@link #LOOP_COUNTER_PARAMETER} in the corresponding parameter
	reference.
<p>
<h4>
	Pattern:
</h4><p>
	PVL source parameter names are matched with map parameter references
	by using case sensitive comparison, ignoring case in the comparision,
	or treating the map parameter reference as a regular expression. Case
	sensitive matching is used by default.
<p>
<h4>
	Verbose/Quiet:
</h4><p>
	The verbose option provides a log of operations, as they occur, to
	the standard output. The quiet option disables the verbose mode.
	By default quiet mode is in effect.
<p>
<h4>
	Unresolved_source_reference_throws:
</h4><p>
	Any unresolved source reference will result in an exception being
	thrown rather than the mapped database field being ignored.
<p>
<h4>
	No-op:
</h4><p>
	As an aid in debugging complex references in map parameter syntax
	final database update operations will be disabled so that the results
	of resolving map parameter references may be viewed without affecting
	the database contents. This mode is implicitly verbose.
<p>
<h4>
	Help:
</h4><p>
	List a command line summary description and immediately exit.
<p>
	<b>N.B.</b>This method always results in a System.exit with a status
	of <code>{@link Update_DB#EXIT_INVALID_COMMAND_LINE_SYNTAX
	EXIT_INVALID_COMMAND_LINE_SYNTAX}</code>.
<p>
	@param	verbose	If true the usage is printed, otherwise silence.
*/
public static void Usage
	(
	boolean	verbose
	)
{
if (verbose)
	System.out.println
		(ID + NL
		+"Usage: PVL_to_DB [<0ptions>] -Map <pathname> <PVL source> [...]" + NL
		+"  Options -" + NL
		+"    -Configuration [<pathname>]"
			+" (default: " + DEFAULT_CONFIGURATION_FILENAME + ")" + NL
		+"    -Set <parameter definition>" + NL
		+"    -Database <server name>"
			+" (default: configuration Server)" + NL
		+"    -Loop [<count>]"
			+" (default: 0, no looping)" + NL
		+"    -Pattern Case_sensitive | Ignore_case | Regex"
			+" (default: case_sensitive)" + NL
		+"    -Quiet | -Verbose"
			+" (default: -quiet)" + NL
		+"    -Unresolved_source_reference_throws"
			+" (default: false)" + NL
		+"    -No-op"
			+" (default: false)" + NL
		+"    -Help" + NL
		);
System.exit (EXIT_INVALID_COMMAND_LINE_SYNTAX);
}

}
