# Copyright 2022 Joao Rodrigues.  All rights reserved.
#
# This file is part of the Biopython distribution and governed by your
# choice of the "Biopython License Agreement" or the "BSD 3-Clause License".
# Please see the LICENSE file that should have been included as part of this
# package.
"""Alphabets used by the wwPDB in structural file formats."""

# Dictionaries in this file were automatically generated from wwPDB data,
# by parsing the ligand expo components dictionary and extracting
# component->parent relationships.
#
# Additional entries to the protein extended set were taken from SCOP.
#
# The 'fmt:off' lines prevent black for formatting the dictionaries.

from Bio.Data.IUPACData import protein_letters_3to1 as _protein_letters_3to1

# Protein
protein_letters_3to1 = {k.upper(): v for k, v in _protein_letters_3to1.items()}
protein_letters_1to3 = {v: k for k, v in protein_letters_3to1.items()}

# fmt: off
protein_letters_3to1_extended = {
    "A5N": "N", "A8E": "V", "A9D": "S", "AA3": "A", "AA4": "A", "AAR": "R",
    "ABA": "A", "ACL": "R", "AEA": "C", "AEI": "D", "AFA": "N", "AGM": "R",
    "AGQ": "Y", "AGT": "C", "AHB": "N", "AHL": "R", "AHO": "A", "AHP": "A",
    "AIB": "A", "AKL": "D", "AKZ": "D", "ALA": "A", "ALC": "A", "ALM": "A",
    "ALN": "A", "ALO": "T", "ALS": "A", "ALT": "A", "ALV": "A", "ALY": "K",
    "AME": "M", "AN6": "L", "AN8": "A", "API": "K", "APK": "K", "AR2": "R",
    "AR4": "E", "AR7": "R", "ARG": "R", "ARM": "R", "ARO": "R", "AS7": "N",
    "ASA": "D", "ASB": "D", "ASI": "D", "ASK": "D", "ASL": "D", "ASN": "N",
    "ASP": "D", "ASQ": "D", "AYA": "A", "AZH": "A", "AZK": "K", "AZS": "S",
    "AZY": "Y", "AVJ": "H", "A30": "Y", "A3U": "F", "ECC": "Q", "ECX": "C",
    "EFC": "C", "EHP": "F", "ELY": "K", "EME": "E", "EPM": "M", "EPQ": "Q",
    "ESB": "Y", "ESC": "M", "EXY": "L", "EXA": "K", "E0Y": "P", "E9V": "H",
    "E9M": "W", "EJA": "C", "EUP": "T", "EZY": "G", "E9C": "Y", "EW6": "S",
    "EXL": "W", "I2M": "I", "I4G": "G", "I58": "K", "IAM": "A", "IAR": "R",
    "ICY": "C", "IEL": "K", "IGL": "G", "IIL": "I", "ILE": "I", "ILG": "E",
    "ILM": "I", "ILX": "I", "ILY": "K", "IML": "I", "IOR": "R", "IPG": "G",
    "IT1": "K", "IYR": "Y", "IZO": "M", "IC0": "G", "M0H": "C", "M2L": "K",
    "M2S": "M", "M30": "G", "M3L": "K", "M3R": "K", "MA ": "A", "MAA": "A",
    "MAI": "R", "MBQ": "Y", "MC1": "S", "MCL": "K", "MCS": "C", "MD3": "C",
    "MD5": "C", "MD6": "G", "MDF": "Y", "ME0": "M", "MEA": "F", "MEG": "E",
    "MEN": "N", "MEQ": "Q", "MET": "M", "MEU": "G", "MFN": "E", "MGG": "R",
    "MGN": "Q", "MGY": "G", "MH1": "H", "MH6": "S", "MHL": "L", "MHO": "M",
    "MHS": "H", "MHU": "F", "MIR": "S", "MIS": "S", "MK8": "L", "ML3": "K",
    "MLE": "L", "MLL": "L", "MLY": "K", "MLZ": "K", "MME": "M", "MMO": "R",
    "MNL": "L", "MNV": "V", "MP8": "P", "MPQ": "G", "MSA": "G", "MSE": "M",
    "MSL": "M", "MSO": "M", "MT2": "M", "MTY": "Y", "MVA": "V", "MYK": "K",
    "MYN": "R", "QCS": "C", "QIL": "I", "QMM": "Q", "QPA": "C", "QPH": "F",
    "Q3P": "K", "QVA": "C", "QX7": "A", "Q2E": "W", "Q75": "M", "Q78": "F",
    "QM8": "L", "QMB": "A", "QNQ": "C", "QNT": "C", "QNW": "C", "QO2": "C",
    "QO5": "C", "QO8": "C", "QQ8": "Q", "U2X": "Y", "U3X": "F", "UF0": "S",
    "UGY": "G", "UM1": "A", "UM2": "A", "UMA": "A", "UQK": "A", "UX8": "W",
    "UXQ": "F", "YCM": "C", "YOF": "Y", "YPR": "P", "YPZ": "Y", "YTH": "T",
    "Y1V": "L", "Y57": "K", "YHA": "K", "200": "F", "23F": "F", "23P": "A",
    "26B": "T", "28X": "T", "2AG": "A", "2CO": "C", "2FM": "M", "2GX": "F",
    "2HF": "H", "2JG": "S", "2KK": "K", "2KP": "K", "2LT": "Y", "2LU": "L",
    "2ML": "L", "2MR": "R", "2MT": "P", "2OR": "R", "2P0": "P", "2QZ": "T",
    "2R3": "Y", "2RA": "A", "2RX": "S", "2SO": "H", "2TY": "Y", "2VA": "V",
    "2XA": "C", "2ZC": "S", "6CL": "K", "6CW": "W", "6GL": "A", "6HN": "K",
    "60F": "C", "66D": "I", "6CV": "A", "6M6": "C", "6V1": "C", "6WK": "C",
    "6Y9": "P", "6DN": "K", "DA2": "R", "DAB": "A", "DAH": "F", "DBS": "S",
    "DBU": "T", "DBY": "Y", "DBZ": "A", "DC2": "C", "DDE": "H", "DDZ": "A",
    "DI7": "Y", "DHA": "S", "DHN": "V", "DIR": "R", "DLS": "K", "DM0": "K",
    "DMH": "N", "DMK": "D", "DNL": "K", "DNP": "A", "DNS": "K", "DNW": "A",
    "DOH": "D", "DON": "L", "DP1": "R", "DPL": "P", "DPP": "A", "DPQ": "Y",
    "DYS": "C", "D2T": "D", "DYA": "D", "DJD": "F", "DYJ": "P", "DV9": "E",
    "H14": "F", "H1D": "M", "H5M": "P", "HAC": "A", "HAR": "R", "HBN": "H",
    "HCM": "C", "HGY": "G", "HHI": "H", "HIA": "H", "HIC": "H", "HIP": "H",
    "HIQ": "H", "HIS": "H", "HL2": "L", "HLU": "L", "HMR": "R", "HNC": "C",
    "HOX": "F", "HPC": "F", "HPE": "F", "HPH": "F", "HPQ": "F", "HQA": "A",
    "HR7": "R", "HRG": "R", "HRP": "W", "HS8": "H", "HS9": "H", "HSE": "S",
    "HSK": "H", "HSL": "S", "HSO": "H", "HT7": "W", "HTI": "C", "HTR": "W",
    "HV5": "A", "HVA": "V", "HY3": "P", "HYI": "M", "HYP": "P", "HZP": "P",
    "HIX": "A", "HSV": "H", "HLY": "K", "HOO": "H", "H7V": "A", "L5P": "K",
    "LRK": "K", "L3O": "L", "LA2": "K", "LAA": "D", "LAL": "A", "LBY": "K",
    "LCK": "K", "LCX": "K", "LDH": "K", "LE1": "V", "LED": "L", "LEF": "L",
    "LEH": "L", "LEM": "L", "LEN": "L", "LET": "K", "LEU": "L", "LEX": "L",
    "LGY": "K", "LLO": "K", "LLP": "K", "LLY": "K", "LLZ": "K", "LME": "E",
    "LMF": "K", "LMQ": "Q", "LNE": "L", "LNM": "L", "LP6": "K", "LPD": "P",
    "LPG": "G", "LPS": "S", "LSO": "K", "LTR": "W", "LVG": "G", "LVN": "V",
    "LWY": "P", "LYF": "K", "LYK": "K", "LYM": "K", "LYN": "K", "LYO": "K",
    "LYP": "K", "LYR": "K", "LYS": "K", "LYU": "K", "LYX": "K", "LYZ": "K",
    "LAY": "L", "LWI": "F", "LBZ": "K", "P1L": "C", "P2Q": "Y", "P2Y": "P",
    "P3Q": "Y", "PAQ": "Y", "PAS": "D", "PAT": "W", "PBB": "C", "PBF": "F",
    "PCA": "Q", "PCC": "P", "PCS": "F", "PE1": "K", "PEC": "C", "PF5": "F",
    "PFF": "F", "PG1": "S", "PGY": "G", "PHA": "F", "PHD": "D", "PHE": "F",
    "PHI": "F", "PHL": "F", "PHM": "F", "PKR": "P", "PLJ": "P", "PM3": "F",
    "POM": "P", "PPN": "F", "PR3": "C", "PR4": "P", "PR7": "P", "PR9": "P",
    "PRJ": "P", "PRK": "K", "PRO": "P", "PRS": "P", "PRV": "G", "PSA": "F",
    "PSH": "H", "PTH": "Y", "PTM": "Y", "PTR": "Y", "PVH": "H", "PXU": "P",
    "PYA": "A", "PYH": "K", "PYX": "C", "PH6": "P", "P9S": "C", "P5U": "S",
    "POK": "R", "T0I": "Y", "T11": "F", "TAV": "D", "TBG": "V", "TBM": "T",
    "TCQ": "Y", "TCR": "W", "TEF": "F", "TFQ": "F", "TH5": "T", "TH6": "T",
    "THC": "T", "THR": "T", "THZ": "R", "TIH": "A", "TIS": "S", "TLY": "K",
    "TMB": "T", "TMD": "T", "TNB": "C", "TNR": "S", "TNY": "T", "TOQ": "W",
    "TOX": "W", "TPJ": "P", "TPK": "P", "TPL": "W", "TPO": "T", "TPQ": "Y",
    "TQI": "W", "TQQ": "W", "TQZ": "C", "TRF": "W", "TRG": "K", "TRN": "W",
    "TRO": "W", "TRP": "W", "TRQ": "W", "TRW": "W", "TRX": "W", "TRY": "W",
    "TS9": "I", "TSY": "C", "TTQ": "W", "TTS": "Y", "TXY": "Y", "TY1": "Y",
    "TY2": "Y", "TY3": "Y", "TY5": "Y", "TY8": "Y", "TY9": "Y", "TYB": "Y",
    "TYC": "Y", "TYE": "Y", "TYI": "Y", "TYJ": "Y", "TYN": "Y", "TYO": "Y",
    "TYQ": "Y", "TYR": "Y", "TYS": "Y", "TYT": "Y", "TYW": "Y", "TYY": "Y",
    "T8L": "T", "T9E": "T", "TNQ": "W", "TSQ": "F", "TGH": "W", "X2W": "E",
    "XCN": "C", "XPR": "P", "XSN": "N", "XW1": "A", "XX1": "K", "XYC": "A",
    "XA6": "F", "11Q": "P", "11W": "E", "12L": "P", "12X": "P", "12Y": "P",
    "143": "C", "1AC": "A", "1L1": "A", "1OP": "Y", "1PA": "F", "1PI": "A",
    "1TQ": "W", "1TY": "Y", "1X6": "S", "56A": "H", "5AB": "A", "5CS": "C",
    "5CW": "W", "5HP": "E", "5OH": "A", "5PG": "G", "51T": "Y", "54C": "W",
    "5CR": "F", "5CT": "K", "5FQ": "A", "5GM": "I", "5JP": "S", "5T3": "K",
    "5MW": "K", "5OW": "K", "5R5": "S", "5VV": "N", "5XU": "A", "55I": "F",
    "999": "D", "9DN": "N", "9NE": "E", "9NF": "F", "9NR": "R", "9NV": "V",
    "9E7": "K", "9KP": "K", "9WV": "A", "9TR": "K", "9TU": "K", "9TX": "K",
    "9U0": "K", "9IJ": "F", "B1F": "F", "B27": "T", "B2A": "A", "B2F": "F",
    "B2I": "I", "B2V": "V", "B3A": "A", "B3D": "D", "B3E": "E", "B3K": "K",
    "B3U": "H", "B3X": "N", "B3Y": "Y", "BB6": "C", "BB7": "C", "BB8": "F",
    "BB9": "C", "BBC": "C", "BCS": "C", "BCX": "C", "BFD": "D", "BG1": "S",
    "BH2": "D", "BHD": "D", "BIF": "F", "BIU": "I", "BL2": "L", "BLE": "L",
    "BLY": "K", "BMT": "T", "BNN": "F", "BOR": "R", "BP5": "A", "BPE": "C",
    "BSE": "S", "BTA": "L", "BTC": "C", "BTK": "K", "BTR": "W", "BUC": "C",
    "BUG": "V", "BYR": "Y", "BWV": "R", "BWB": "S", "BXT": "S", "F2F": "F",
    "F2Y": "Y", "FAK": "K", "FB5": "A", "FB6": "A", "FC0": "F", "FCL": "F",
    "FDL": "K", "FFM": "C", "FGL": "G", "FGP": "S", "FH7": "K", "FHL": "K",
    "FHO": "K", "FIO": "R", "FLA": "A", "FLE": "L", "FLT": "Y", "FME": "M",
    "FOE": "C", "FP9": "P", "FPK": "P", "FT6": "W", "FTR": "W", "FTY": "Y",
    "FVA": "V", "FZN": "K", "FY3": "Y", "F7W": "W", "FY2": "Y", "FQA": "K",
    "F7Q": "Y", "FF9": "K", "FL6": "D", "JJJ": "C", "JJK": "C", "JJL": "C",
    "JLP": "K", "J3D": "C", "J9Y": "R", "J8W": "S", "JKH": "P", "N10": "S",
    "N7P": "P", "NA8": "A", "NAL": "A", "NAM": "A", "NBQ": "Y", "NC1": "S",
    "NCB": "A", "NEM": "H", "NEP": "H", "NFA": "F", "NIY": "Y", "NLB": "L",
    "NLE": "L", "NLN": "L", "NLO": "L", "NLP": "L", "NLQ": "Q", "NLY": "G",
    "NMC": "G", "NMM": "R", "NNH": "R", "NOT": "L", "NPH": "C", "NPI": "A",
    "NTR": "Y", "NTY": "Y", "NVA": "V", "NWD": "A", "NYB": "C", "NYS": "C",
    "NZH": "H", "N80": "P", "NZC": "T", "NLW": "L", "N0A": "F", "N9P": "A",
    "N65": "K", "R1A": "C", "R4K": "W", "RE0": "W", "RE3": "W", "RGL": "R",
    "RGP": "E", "RT0": "P", "RVX": "S", "RZ4": "S", "RPI": "R", "RVJ": "A",
    "VAD": "V", "VAF": "V", "VAH": "V", "VAI": "V", "VAL": "V", "VB1": "K",
    "VH0": "P", "VR0": "R", "V44": "C", "V61": "F", "VPV": "K", "V5N": "H",
    "V7T": "K", "Z01": "A", "Z3E": "T", "Z70": "H", "ZBZ": "C", "ZCL": "F",
    "ZU0": "T", "ZYJ": "P", "ZYK": "P", "ZZD": "C", "ZZJ": "A", "ZIQ": "W",
    "ZPO": "P", "ZDJ": "Y", "ZT1": "K", "30V": "C", "31Q": "C", "33S": "F",
    "33W": "A", "34E": "V", "3AH": "H", "3BY": "P", "3CF": "F", "3CT": "Y",
    "3GA": "A", "3GL": "E", "3MD": "D", "3MY": "Y", "3NF": "Y", "3O3": "E",
    "3PX": "P", "3QN": "K", "3TT": "P", "3XH": "G", "3YM": "Y", "3WS": "A",
    "3WX": "P", "3X9": "C", "3ZH": "H", "7JA": "I", "73C": "S", "73N": "R",
    "73O": "Y", "73P": "K", "74P": "K", "7N8": "F", "7O5": "A", "7XC": "F",
    "7ID": "D", "7OZ": "A", "C1S": "C", "C1T": "C", "C1X": "K", "C22": "A",
    "C3Y": "C", "C4R": "C", "C5C": "C", "C6C": "C", "CAF": "C", "CAS": "C",
    "CAY": "C", "CCS": "C", "CEA": "C", "CGA": "E", "CGU": "E", "CGV": "C",
    "CHP": "G", "CIR": "R", "CLE": "L", "CLG": "K", "CLH": "K", "CME": "C",
    "CMH": "C", "CML": "C", "CMT": "C", "CR5": "G", "CS0": "C", "CS1": "C",
    "CS3": "C", "CS4": "C", "CSA": "C", "CSB": "C", "CSD": "C", "CSE": "C",
    "CSJ": "C", "CSO": "C", "CSP": "C", "CSR": "C", "CSS": "C", "CSU": "C",
    "CSW": "C", "CSX": "C", "CSZ": "C", "CTE": "W", "CTH": "T", "CWD": "A",
    "CWR": "S", "CXM": "M", "CY0": "C", "CY1": "C", "CY3": "C", "CY4": "C",
    "CYA": "C", "CYD": "C", "CYF": "C", "CYG": "C", "CYJ": "K", "CYM": "C",
    "CYQ": "C", "CYR": "C", "CYS": "C", "CYW": "C", "CZ2": "C", "CZZ": "C",
    "CG6": "C", "C1J": "R", "C4G": "R", "C67": "R", "C6D": "R", "CE7": "N",
    "CZS": "A", "G01": "E", "G8M": "E", "GAU": "E", "GEE": "G", "GFT": "S",
    "GHC": "E", "GHG": "Q", "GHW": "E", "GL3": "G", "GLH": "Q", "GLJ": "E",
    "GLK": "E", "GLN": "Q", "GLQ": "E", "GLU": "E", "GLY": "G", "GLZ": "G",
    "GMA": "E", "GME": "E", "GNC": "Q", "GPL": "K", "GSC": "G", "GSU": "E",
    "GT9": "C", "GVL": "S", "G3M": "R", "G5G": "L", "G1X": "Y", "G8X": "P",
    "K1R": "C", "KBE": "K", "KCX": "K", "KFP": "K", "KGC": "K", "KNB": "A",
    "KOR": "M", "KPI": "K", "KPY": "K", "KST": "K", "KYN": "W", "KYQ": "K",
    "KCR": "K", "KPF": "K", "K5L": "S", "KEO": "K", "KHB": "K", "KKD": "D",
    "K5H": "C", "K7K": "S", "OAR": "R", "OAS": "S", "OBS": "K", "OCS": "C",
    "OCY": "C", "OHI": "H", "OHS": "D", "OLD": "H", "OLT": "T", "OLZ": "S",
    "OMH": "S", "OMT": "M", "OMX": "Y", "OMY": "Y", "ONH": "A", "ORN": "A",
    "ORQ": "R", "OSE": "S", "OTH": "T", "OXX": "D", "OYL": "H", "O7A": "T",
    "O7D": "W", "O7G": "V", "O2E": "S", "O6H": "W", "OZW": "F", "S12": "S",
    "S1H": "S", "S2C": "C", "S2P": "A", "SAC": "S", "SAH": "C", "SAR": "G",
    "SBG": "S", "SBL": "S", "SCH": "C", "SCS": "C", "SCY": "C", "SD4": "N",
    "SDB": "S", "SDP": "S", "SEB": "S", "SEE": "S", "SEG": "A", "SEL": "S",
    "SEM": "S", "SEN": "S", "SEP": "S", "SER": "S", "SET": "S", "SGB": "S",
    "SHC": "C", "SHP": "G", "SHR": "K", "SIB": "C", "SLL": "K", "SLZ": "K",
    "SMC": "C", "SME": "M", "SMF": "F", "SNC": "C", "SNN": "N", "SOY": "S",
    "SRZ": "S", "STY": "Y", "SUN": "S", "SVA": "S", "SVV": "S", "SVW": "S",
    "SVX": "S", "SVY": "S", "SVZ": "S", "SXE": "S", "SKH": "K", "SNM": "S",
    "SNK": "H", "SWW": "S", "WFP": "F", "WLU": "L", "WPA": "F", "WRP": "W",
    "WVL": "V", "02K": "A", "02L": "N", "02O": "A", "02Y": "A", "033": "V",
    "037": "P", "03Y": "C", "04U": "P", "04V": "P", "05N": "P", "07O": "C",
    "0A0": "D", "0A1": "Y", "0A2": "K", "0A8": "C", "0A9": "F", "0AA": "V",
    "0AB": "V", "0AC": "G", "0AF": "W", "0AG": "L", "0AH": "S", "0AK": "D",
    "0AR": "R", "0BN": "F", "0CS": "A", "0E5": "T", "0EA": "Y", "0FL": "A",
    "0LF": "P", "0NC": "A", "0PR": "Y", "0QL": "C", "0TD": "D", "0UO": "W",
    "0WZ": "Y", "0X9": "R", "0Y8": "P", "4AF": "F", "4AR": "R", "4AW": "W",
    "4BF": "Y", "4CF": "F", "4CY": "M", "4DP": "W", "4FB": "P", "4FW": "W",
    "4HL": "Y", "4HT": "W", "4IN": "W", "4MM": "M", "4PH": "F", "4U7": "A",
    "41H": "F", "41Q": "N", "42Y": "S", "432": "S", "45F": "P", "4AK": "K",
    "4D4": "R", "4GJ": "C", "4KY": "P", "4L0": "P", "4LZ": "Y", "4N7": "P",
    "4N8": "P", "4N9": "P", "4OG": "W", "4OU": "F", "4OV": "S", "4OZ": "S",
    "4PQ": "W", "4SJ": "F", "4WQ": "A", "4HH": "S", "4HJ": "S", "4J4": "C",
    "4J5": "R", "4II": "F", "4VI": "R", "823": "N", "8SP": "S", "8AY": "A",
}

# Nucleic Acids
nucleic_letters_3to1 = {
    "A  ": "A", "C  ": "C", "G  ": "G", "U  ": "U",
    "DA ": "A", "DC ": "C", "DG ": "G", "DT ": "T",
}

# fmt: off
nucleic_letters_3to1_extended = {
    "A  ": "A", "A23": "A", "A2L": "A", "A2M": "A", "A34": "A", "A35": "A",
    "A38": "A", "A39": "A", "A3A": "A", "A3P": "A", "A40": "A", "A43": "A",
    "A44": "A", "A47": "A", "A5L": "A", "A5M": "C", "A5O": "A", "A6A": "A",
    "A6C": "C", "A6G": "G", "A6U": "U", "A7E": "A", "A9Z": "A", "ABR": "A",
    "ABS": "A", "AD2": "A", "ADI": "A", "ADP": "A", "AET": "A", "AF2": "A",
    "AFG": "G", "AMD": "A", "AMO": "A", "AP7": "A", "AS ": "A", "ATD": "T",
    "ATL": "T", "ATM": "T", "AVC": "A", "AI5": "C", "E  ": "A", "E1X": "A",
    "EDA": "A", "EFG": "G", "EHG": "G", "EIT": "T", "EXC": "C", "E3C": "C",
    "E6G": "G", "E7G": "G", "EQ4": "G", "EAN": "T", "I5C": "C", "IC ": "C",
    "IG ": "G", "IGU": "G", "IMC": "C", "IMP": "G", "IU ": "U", "I4U": "U",
    "IOO": "G", "M1G": "G", "M2G": "G", "M4C": "C", "M5M": "C", "MA6": "A",
    "MA7": "A", "MAD": "A", "MCY": "C", "ME6": "C", "MEP": "U", "MG1": "G",
    "MGQ": "A", "MGT": "G", "MGV": "G", "MIA": "A", "MMT": "T", "MNU": "U",
    "MRG": "G", "MTR": "T", "MTU": "A", "MFO": "G", "M7A": "A", "MHG": "G",
    "MMX": "C", "QUO": "G", "QCK": "T", "QSQ": "A", "U  ": "U", "U25": "U",
    "U2L": "U", "U2P": "U", "U31": "U", "U34": "U", "U36": "U", "U37": "U",
    "U8U": "U", "UAR": "U", "UBB": "U", "UBD": "U", "UD5": "U", "UPV": "U",
    "UR3": "U", "URD": "U", "US3": "T", "US5": "U", "UZR": "U", "UMO": "U",
    "U23": "U", "U48": "C", "U7B": "C", "Y  ": "A", "YCO": "C", "YG ": "G",
    "YYG": "G", "23G": "G", "26A": "A", "2AR": "A", "2AT": "T", "2AU": "U",
    "2BT": "T", "2BU": "A", "2DA": "A", "2DT": "T", "2EG": "G", "2GT": "T",
    "2JV": "G", "2MA": "A", "2MG": "G", "2MU": "U", "2NT": "T", "2OM": "U",
    "2OT": "T", "2PR": "G", "2SG": "G", "2ST": "T", "63G": "G", "63H": "G",
    "64T": "T", "68Z": "G", "6CT": "T", "6HA": "A", "6HB": "A", "6HC": "C",
    "6HG": "G", "6HT": "T", "6IA": "A", "6MA": "A", "6MC": "A", "6MP": "A",
    "6MT": "A", "6MZ": "A", "6OG": "G", "6PO": "G", "6FK": "G", "6NW": "A",
    "6OO": "C", "D00": "C", "D3T": "T", "D4M": "T", "DA ": "A", "DC ": "C",
    "DCG": "G", "DCT": "C", "DDG": "G", "DFC": "C", "DFG": "G", "DG ": "G",
    "DG8": "G", "DGI": "G", "DGP": "G", "DHU": "U", "DNR": "C", "DOC": "C",
    "DPB": "T", "DRT": "T", "DT ": "T", "DZM": "A", "D4B": "C", "H2U": "U",
    "HN0": "G", "HN1": "G", "LC ": "C", "LCA": "A", "LCG": "G", "LG ": "G",
    "LGP": "G", "LHU": "U", "LSH": "T", "LST": "T", "LDG": "G", "L3X": "A",
    "LHH": "C", "LV2": "C", "L1J": "G", "P  ": "G", "P2T": "T", "P5P": "A",
    "PG7": "G", "PGN": "G", "PGP": "G", "PMT": "C", "PPU": "A", "PPW": "G",
    "PR5": "A", "PRN": "A", "PST": "T", "PSU": "U", "PU ": "A", "PVX": "C",
    "PYO": "U", "PZG": "G", "P4U": "U", "P7G": "G", "T  ": "T", "T2S": "T",
    "T31": "U", "T32": "T", "T36": "T", "T37": "T", "T38": "T", "T39": "T",
    "T3P": "T", "T41": "T", "T48": "T", "T49": "T", "T4S": "T", "T5S": "T",
    "T64": "T", "T6A": "A", "TA3": "T", "TAF": "T", "TBN": "A", "TC1": "C",
    "TCP": "T", "TCY": "A", "TDY": "T", "TED": "T", "TFE": "T", "TFF": "T",
    "TFO": "A", "TFT": "T", "TGP": "G", "TCJ": "C", "TLC": "T", "TP1": "T",
    "TPC": "C", "TPG": "G", "TSP": "T", "TTD": "T", "TTM": "T", "TXD": "A",
    "TXP": "A", "TC ": "C", "TG ": "G", "T0N": "G", "T0Q": "G", "X  ": "G",
    "XAD": "A", "XAL": "A", "XCL": "C", "XCR": "C", "XCT": "C", "XCY": "C",
    "XGL": "G", "XGR": "G", "XGU": "G", "XPB": "G", "XTF": "T", "XTH": "T",
    "XTL": "T", "XTR": "T", "XTS": "G", "XUA": "A", "XUG": "G", "102": "G",
    "10C": "C", "125": "U", "126": "U", "127": "U", "12A": "A", "16B": "C",
    "18M": "G", "1AP": "A", "1CC": "C", "1FC": "C", "1MA": "A", "1MG": "G",
    "1RN": "U", "1SC": "C", "5AA": "A", "5AT": "T", "5BU": "U", "5CG": "G",
    "5CM": "C", "5FA": "A", "5FC": "C", "5FU": "U", "5HC": "C", "5HM": "C",
    "5HT": "T", "5IC": "C", "5IT": "T", "5MC": "C", "5MU": "U", "5NC": "C",
    "5PC": "C", "5PY": "T", "9QV": "U", "94O": "T", "9SI": "A", "9SY": "A",
    "B7C": "C", "BGM": "G", "BOE": "T", "B8H": "U", "B8K": "G", "B8Q": "C",
    "B8T": "C", "B8W": "G", "B9B": "G", "B9H": "C", "BGH": "G", "F3H": "T",
    "F3N": "A", "F4H": "T", "FA2": "A", "FDG": "G", "FHU": "U", "FMG": "G",
    "FNU": "U", "FOX": "G", "F2T": "U", "F74": "G", "F4Q": "G", "F7H": "C",
    "F7K": "G", "JDT": "T", "JMH": "C", "J0X": "C", "N5M": "C", "N6G": "G",
    "N79": "A", "NCU": "C", "NMS": "T", "NMT": "T", "NTT": "T", "N7X": "C",
    "R  ": "A", "RBD": "A", "RDG": "G", "RIA": "A", "RMP": "A", "RPC": "C",
    "RSP": "C", "RSQ": "C", "RT ": "T", "RUS": "U", "RFJ": "G", "V3L": "A",
    "VC7": "G", "Z  ": "C", "ZAD": "A", "ZBC": "C", "ZBU": "U", "ZCY": "C",
    "ZGU": "G", "31H": "A", "31M": "A", "3AU": "U", "3DA": "A", "3ME": "U",
    "3MU": "U", "3TD": "U", "70U": "U", "7AT": "A", "7DA": "A", "7GU": "G",
    "7MG": "G", "7BG": "G", "73W": "C", "75B": "U", "7OK": "C", "7S3": "G",
    "7SN": "G", "C  ": "C", "C25": "C", "C2L": "C", "C2S": "C", "C31": "C",
    "C32": "C", "C34": "C", "C36": "C", "C37": "C", "C38": "C", "C42": "C",
    "C43": "C", "C45": "C", "C46": "C", "C49": "C", "C4S": "C", "C5L": "C",
    "C6G": "G", "CAR": "C", "CB2": "C", "CBR": "C", "CBV": "C", "CCC": "C",
    "CDW": "C", "CFL": "C", "CFZ": "C", "CG1": "G", "CH ": "C", "CMR": "C",
    "CNU": "U", "CP1": "C", "CSF": "C", "CSL": "C", "CTG": "T", "CX2": "C",
    "C7S": "C", "C7R": "C", "G  ": "G", "G1G": "G", "G25": "G", "G2L": "G",
    "G2S": "G", "G31": "G", "G32": "G", "G33": "G", "G36": "G", "G38": "G",
    "G42": "G", "G46": "G", "G47": "G", "G48": "G", "G49": "G", "G7M": "G",
    "GAO": "G", "GCK": "C", "GDO": "G", "GDP": "G", "GDR": "G", "GF2": "G",
    "GFL": "G", "GH3": "G", "GMS": "G", "GN7": "G", "GNG": "G", "GOM": "G",
    "GRB": "G", "GS ": "G", "GSR": "G", "GSS": "G", "GTP": "G", "GX1": "G",
    "KAG": "G", "KAK": "G", "O2G": "G", "OGX": "G", "OMC": "C", "OMG": "G",
    "OMU": "U", "ONE": "U", "O2Z": "A", "OKN": "C", "OKQ": "C", "S2M": "T",
    "S4A": "A", "S4C": "C", "S4G": "G", "S4U": "U", "S6G": "G", "SC ": "C",
    "SDE": "A", "SDG": "G", "SDH": "G", "SMP": "A", "SMT": "T", "SPT": "T",
    "SRA": "A", "SSU": "U", "SUR": "U", "00A": "A", "0AD": "G", "0AM": "A",
    "0AP": "C", "0AV": "A", "0R8": "C", "0SP": "A", "0UH": "G", "47C": "C",
    "4OC": "C", "4PC": "C", "4PD": "C", "4PE": "C", "4SC": "C", "4SU": "U",
    "45A": "A", "4U3": "C", "8AG": "G", "8AN": "A", "8BA": "A", "8FG": "G",
    "8MG": "G", "8OG": "G", "8PY": "G", "8AA": "G", "85Y": "U", "8OS": "G",
}

# Solvent accesibility scales
residue_sasa_scales = {
    # Ahmad: Ahmad et al. 2003 https://doi.org/10.1002/prot.10328
    "Ahmad": {
        "ALA": 110.2,
        "ARG": 229.0,
        "ASN": 146.4,
        "ASP": 144.1,
        "CYS": 140.4,
        "GLN": 178.6,
        "GLU": 174.7,
        "GLY": 78.7,
        "HIS": 181.9,
        "ILE": 183.1,
        "LEU": 164.0,
        "LYS": 205.7,
        "MET": 200.1,
        "PHE": 200.7,
        "PRO": 141.9,
        "SER": 117.2,
        "THR": 138.7,
        "TRP": 240.5,
        "TYR": 213.7,
        "VAL": 153.7,
    },
    # Miller max acc: Miller et al. 1987 https://doi.org/10.1016/0022-2836(87)90038-6
    "Miller": {
        "ALA": 113.0,
        "ARG": 241.0,
        "ASN": 158.0,
        "ASP": 151.0,
        "CYS": 140.0,
        "GLN": 189.0,
        "GLU": 183.0,
        "GLY": 85.0,
        "HIS": 194.0,
        "ILE": 182.0,
        "LEU": 180.0,
        "LYS": 211.0,
        "MET": 204.0,
        "PHE": 218.0,
        "PRO": 143.0,
        "SER": 122.0,
        "THR": 146.0,
        "TRP": 259.0,
        "TYR": 229.0,
        "VAL": 160.0,
    },
    # Sander: Sander & Rost 1994 https://doi.org/10.1002/prot.340200303
    "Sander": {
        "ALA": 106.0,
        "ARG": 248.0,
        "ASN": 157.0,
        "ASP": 163.0,
        "CYS": 135.0,
        "GLN": 198.0,
        "GLU": 194.0,
        "GLY": 84.0,
        "HIS": 184.0,
        "ILE": 169.0,
        "LEU": 164.0,
        "LYS": 205.0,
        "MET": 188.0,
        "PHE": 197.0,
        "PRO": 136.0,
        "SER": 130.0,
        "THR": 142.0,
        "TRP": 227.0,
        "TYR": 222.0,
        "VAL": 142.0,
    },
    # Wilke: Tien et al. 2013 https://doi.org/10.1371/journal.pone.0080635
    "Wilke": {
        "ALA": 129.0,
        "ARG": 274.0,
        "ASN": 195.0,
        "ASP": 193.0,
        "CYS": 167.0,
        "GLN": 225.0,
        "GLU": 223.0,
        "GLY": 104.0,
        "HIS": 224.0,
        "ILE": 197.0,
        "LEU": 201.0,
        "LYS": 236.0,
        "MET": 224.0,
        "PHE": 240.0,
        "PRO": 159.0,
        "SER": 155.0,
        "THR": 172.0,
        "TRP": 285.0,
        "TYR": 263.0,
        "VAL": 174.0,
    }
}
