/*************************************************
*     rpld - an IBM style RIPL server            *
*************************************************/

/*
 * bpf support code
 * Copyright (C) 2001 YAMAMOTO Takashi <yamt@netbsd.org>.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENCE file which can be found at the top level of
 * the rpld distribution.
 */

static char rcsid[] =
  "$Id: bpf-nit.c,v 1.5 2001/11/01 15:30:35 root Exp root $";

/*
 * $Log: bpf-nit.c,v $
 * Revision 1.5  2001/11/01 15:30:35  root
 * #
 *
 * Revision 1.4  2001/11/01 15:28:23  root
 * #
 *
 * Revision 1.3  2001/11/01 15:28:07  root
 * #
 *
 * Revision 1.2  2001/11/01 15:26:29  root
 * #
 *
 * Revision 1.1  2001/11/01 15:24:26  root
 * #
 *
 *
 */

#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/time.h>

#include <net/bpf.h>
#include <net/if.h>
#include <net/if_dl.h>
#include <net/if_ether.h>

#include <errno.h>
#include <fcntl.h>
#include <ifaddrs.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "project.h"
#include "nit.h"

#define BPF "/dev/bpf"
#ifndef min
#define min(a,b) (((a)<(b))?(a):(b))
#endif

static int bpf_open (const char *);

struct nit
{
  int fd;
  unsigned char lladdr[ETHER_ADDR_LEN];
  size_t bufsize;
  unsigned char *buf;
};

static void
get_hwaddr (unsigned char *name, unsigned char *addr)
{
  struct ifaddrs *ifa0, *ifa;
  struct sockaddr_dl *sdl = 0;

  if (getifaddrs (&ifa0) == -1)
    {
      syslog (LOG_ERR, "get_hwaddr");
      return;
    }

  for (ifa = ifa0; ifa; ifa = ifa->ifa_next)
    {
      if (ifa->ifa_addr->sa_family == AF_LINK)
        {
          if (!strcmp (ifa->ifa_name, (char *) name))
            {
              sdl = (void *) ifa->ifa_addr;
            }
        }
    }

  if (sdl && sdl->sdl_alen == ETHER_ADDR_LEN)
    memcpy (addr, LLADDR (sdl), ETHER_ADDR_LEN);
  else
    syslog (LOG_ERR, "can't get hwaddr of %s", name);

  freeifaddrs (ifa0);
}

void
nit_close (struct nit *n)
{

  if (!n)
    return;

  close (n->fd);
  free (n->buf);
  free (n);
}

struct nit *
nit_open (char *name)
{
  struct nit *nit = 0;

  if (strlen (name) >= IFNAMSIZ)
    {
      syslog (LOG_ERR, "nit_open: too long if name");
      goto err;
    }

  nit = malloc (sizeof (*nit));
  if (!nit)
    {
      syslog (LOG_ERR, "nit_open: %s", strerror (errno));
      goto err;
    }

  nit->bufsize = BPF_MAXBUFSIZE;
  nit->buf = malloc (nit->bufsize);
  if (!nit->buf)
    {
      syslog (LOG_ERR, "nit_open: %s", strerror (errno));
      goto err;
    }

  nit->fd = bpf_open (name);
  if (nit->fd == -1)
    goto err;
  get_hwaddr (name, nit->lladdr);

  return nit;
err:
  if (nit)
    free (nit);

  return 0;
}

unsigned char *
nit_mac (struct nit *n)
{

  return n->lladdr;
}

int
nit_send (struct nit *n, unsigned char *frame, int len, unsigned char *to)
{
  struct ether_header eh;
  struct iovec iov[2];
  ssize_t size;

  memset (&eh, 0, sizeof (eh));
  memcpy (eh.ether_dhost, to, ETHER_ADDR_LEN);
  memcpy (eh.ether_shost, n->lladdr, ETHER_ADDR_LEN);

  iov[0].iov_base = &eh;
  iov[0].iov_len = sizeof (eh);
  iov[1].iov_base = frame;
  iov[1].iov_len = len;

  size = writev (n->fd, iov, 2);
  if (size != len + sizeof (eh))
    {
      syslog (LOG_ERR, "nit_send: %s", strerror (errno));
      return -1;
    }

  return len;
}

 /*ARGSUSED*/ int
nit_multicast (struct nit *n, unsigned char *mcaddr)
{
  u_int flag = 1;

  /* XXX */
  if (ioctl (n->fd, BIOCPROMISC, &flag) == -1)
    {
      syslog (1, "BIOCPROMISC: %s", strerror (errno));
      return -1;
    }

  return 0;
}

int
nit_recv (struct nit *n, unsigned char *buf, int len,
          unsigned char *ufrom, struct timeval *tv)
{
  fd_set fds;
  ssize_t size;
  int r;
  struct bpf_hdr *bh;
  struct ether_header *eh;
  unsigned char *p;
  u_int16_t etype;

  FD_ZERO (&fds);
  FD_SET (n->fd, &fds);

  for (;;)
    {
      r = select (n->fd + 1, &fds, 0, 0, tv);
      if (r > 0)
        break;
      if (r == 0)
        return 0;
      if (r == -1 && errno == EINTR)
        continue;
      syslog (LOG_ERR, "nit_recv: select: %s", strerror (errno));
      return -1;
    }

  if (FD_ISSET (n->fd, &fds))
    {
      size = read (n->fd, n->buf, n->bufsize);
      if (size == -1)
        {
          syslog (LOG_ERR, "read: %s", strerror (errno));
          return -1;
        }
    }
  else
    return 0;                   /* XXX */

  bh = (void *) n->buf;
  size = min (len, bh->bh_datalen - ETHER_HDR_LEN);
  p = n->buf + bh->bh_hdrlen;
  eh = (void *) p;
  p += ETHER_HDR_LEN;
  if (!memcmp (eh->ether_shost, n->lladdr, ETHER_ADDR_LEN))
    return 0;                   /* myself */
  etype = ntohs (eh->ether_type);
  if (etype > ETHERMTU || etype < ETHERMIN)
    return 0;
  size = min (size, etype);
  memcpy (ufrom, eh->ether_shost, ETHER_ADDR_LEN);
  memcpy (buf, p, size);

  return size;
}

int
bpf_open (const char *ifname)
{
  int fd;
  int i, n;
  char filename[PATH_MAX + 1];
  u_int bufsize;
  u_int flag;
  struct ifreq ifr;

  for (i = 0; i < 9; i++)
    {
      snprintf (filename, sizeof (filename), BPF "%u", i);
      fd = open (filename, O_RDWR, 0);
      if (fd == -1)
        {
          if (errno == EBUSY)
            continue;
          break;
        }
      else
        break;
    }

  if (fd == -1)
    {
      syslog (1, "bpf_open: %s", filename);
      goto err;
    }

  bufsize = BPF_MAXBUFSIZE;
  if (ioctl (fd, BIOCSBLEN, &bufsize) == -1)
    syslog (LOG_WARNING, "BIOCSBLEN: %s", strerror (errno));

  memset (&ifr, 0, sizeof (ifr));
  strlcpy (ifr.ifr_name, ifname, sizeof (ifr.ifr_name));
  if (ioctl (fd, BIOCSETIF, &ifr) == -1)
    {
      syslog (LOG_ERR, "BIOCSETIF: %s: %s", ifname, strerror (errno));
      goto err;
    }

  flag = 1;
  if (ioctl (fd, BIOCIMMEDIATE, &flag) == -1)
    {
      syslog (LOG_ERR, "BIOCIMMEDIATE: %s", strerror (errno));
      goto err;
    }

  flag = 1;
  if (ioctl (fd, BIOCSHDRCMPLT, &flag) == -1)
    {
      syslog (LOG_ERR, "BIOCSHDRCMPLT: %s", strerror (errno));
      goto err;
    }

  return fd;
err:
  if (fd != -1)
    close (fd);

  return -1;
}
